import { PrismaClient } from '@prisma/client'
import OpenAI from 'openai'

const prisma = new PrismaClient()

// OpenAI client
const openai = new OpenAI({
  apiKey: process.env.OPENAI_API_KEY,
})

// Popüler ligler ve takımlar
const leagues = [
  {
    name: 'Premier League',
    matches: [
      { home: 'Manchester City', away: 'Arsenal' },
      { home: 'Liverpool', away: 'Chelsea' },
      { home: 'Manchester United', away: 'Tottenham' },
      { home: 'Newcastle', away: 'Brighton' },
      { home: 'Aston Villa', away: 'West Ham' },
    ]
  },
  {
    name: 'La Liga',
    matches: [
      { home: 'Real Madrid', away: 'Barcelona' },
      { home: 'Atletico Madrid', away: 'Sevilla' },
      { home: 'Real Sociedad', away: 'Valencia' },
      { home: 'Villarreal', away: 'Athletic Bilbao' },
    ]
  },
  {
    name: 'Süper Lig',
    matches: [
      { home: 'Fenerbahçe', away: 'Galatasaray' },
      { home: 'Beşiktaş', away: 'Trabzonspor' },
      { home: 'Başakşehir', away: 'Kasımpaşa' },
      { home: 'Antalyaspor', away: 'Konyaspor' },
    ]
  },
  {
    name: 'Serie A',
    matches: [
      { home: 'Inter', away: 'AC Milan' },
      { home: 'Juventus', away: 'Roma' },
      { home: 'Napoli', away: 'Lazio' },
    ]
  },
  {
    name: 'Bundesliga',
    matches: [
      { home: 'Bayern Munich', away: 'Dortmund' },
      { home: 'RB Leipzig', away: 'Bayer Leverkusen' },
      { home: 'Frankfurt', away: 'Wolfsburg' },
    ]
  }
]

// Rastgele maç seç
function getRandomMatches(count: number) {
  const allMatches: any[] = []
  
  leagues.forEach(league => {
    league.matches.forEach(match => {
      allMatches.push({
        ...match,
        league: league.name
      })
    })
  })

  // Rastgele karıştır
  const shuffled = allMatches.sort(() => 0.5 - Math.random())
  return shuffled.slice(0, count)
}

// AI ile tahmin al
async function getAIPrediction(homeTeam: string, awayTeam: string, league: string) {
  try {
    const prompt = `Sen profesyonel bir iddaa analisti ve tahmin uzmanısın. Aşağıdaki maç için SADECE yüksek güvenilirlikli tahmin yap:

Maç: ${homeTeam} vs ${awayTeam}
Lig: ${league}

Lütfen şu formatta yanıt ver (JSON):
{
  "prediction": "Tahmin (örn: 1, X, 2, Üst 2.5, Alt 2.5, KG Var, vb.)",
  "odds": "Tahmini oran (1.50 ile 2.50 arası gerçekçi)",
  "confidence": "Sadece Yüksek yazabilirsin"
}

Not: SADECE yüksek güvenilirlikli ve gerçekçi tahmin ver. Ortalama 1.60-2.20 arası oranlar tercih et.`

    const completion = await openai.chat.completions.create({
      model: 'gpt-4o-mini',
      messages: [
        {
          role: 'system',
          content: 'Sen profesyonel iddaa analistisin. SADECE yüksek güvenilirlikli tahminler veriyorsun. Yanıtların JSON formatında.'
        },
        {
          role: 'user',
          content: prompt
        }
      ],
      temperature: 0.7,
      max_tokens: 200,
      response_format: { type: "json_object" }
    })

    const result = JSON.parse(completion.choices[0].message.content || '{}')
    return result
  } catch (error) {
    console.error(`AI tahmin hatası (${homeTeam} - ${awayTeam}):`, error)
    // Fallback - manuel tahmin
    return {
      prediction: '1',
      odds: '1.85',
      confidence: 'Yüksek'
    }
  }
}

// Günlük kuponları oluştur
async function generateDailyCoupons() {
  try {
    console.log('🤖 İddaaSohbet AI - Günlük Kupon Oluşturuluyor...\n')

    // Admin kullanıcıyı bul
    const adminUser = await prisma.user.findFirst({
      where: { role: 'ADMIN' },
      select: { id: true }
    })

    if (!adminUser) {
      console.error('❌ Admin kullanıcı bulunamadı!')
      return
    }

    // 3 kupon oluştur
    for (let couponIndex = 1; couponIndex <= 3; couponIndex++) {
      console.log(`\n📋 Kupon ${couponIndex}/3 oluşturuluyor...`)

      // Her kupon için 3 maç seç
      const selectedMatches = getRandomMatches(3)
      const matches: any[] = []
      let totalOdds = 1

      for (const match of selectedMatches) {
        console.log(`  🔄 ${match.home} - ${match.away} analiz ediliyor...`)
        
        const aiPrediction = await getAIPrediction(match.home, match.away, match.league)
        
        const odds = parseFloat(aiPrediction.odds)
        totalOdds *= odds

        matches.push({
          homeTeam: match.home,
          awayTeam: match.away,
          league: match.league,
          prediction: aiPrediction.prediction,
          odds: odds,
          category: 'FUTBOL',
          matchDate: new Date(Date.now() + 24 * 60 * 60 * 1000) // Yarın
        })

        console.log(`  ✅ Tahmin: ${aiPrediction.prediction} (${odds})`)
      }

      // Kupon başlığı oluştur
      const titles = [
        `🔥 Günün Banko ${couponIndex}. Kuponu`,
        `⚡ İddaaSohbet AI Önerisi #${couponIndex}`,
        `🏆 Yüksek Oranlı Seçim ${couponIndex}`
      ]

      const coupon = await prisma.coupon.create({
        data: {
          title: titles[couponIndex - 1],
          description: `İddaaSohbet AI tarafından analiz edilmiş ${matches.length} maçlık profesyonel kupon. Yüksek kazanma olasılıklı seçimler.`,
          totalOdds: parseFloat(totalOdds.toFixed(2)),
          stake: 100,
          potentialWin: parseFloat((totalOdds * 100).toFixed(2)),
          status: 'PENDING',
          userId: adminUser.id,
          aiGenerated: true,
          matches: {
            create: matches
          }
        },
        include: {
          matches: true
        }
      })

      console.log(`✅ Kupon oluşturuldu! ID: ${coupon.id}`)
      console.log(`   Toplam Oran: ${coupon.totalOdds}`)
      console.log(`   Maçlar: ${matches.length}`)
    }

    console.log('\n✅ 3 günlük kupon başarıyla oluşturuldu!')
    console.log('🌐 Ana sayfada görüntülenecek!')
  } catch (error) {
    console.error('❌ Hata:', error)
  } finally {
    await prisma.$disconnect()
  }
}

// Çalıştır
generateDailyCoupons()

