/**
 * LICENSE KONTROL SCRIPT'İ
 * 
 * Mevcut license durumunu kontrol eder
 * 
 * KULLANIM:
 * npx tsx scripts/check-license.ts
 */

import { PrismaClient } from '@prisma/client'
import { decryptDomain, verifyLicenseHash } from '../lib/license'

const prisma = new PrismaClient()

async function checkLicense() {
  console.log('\n🔍 LICENSE DURUMU KONTROL EDİLİYOR...\n')
  
  try {
    const license = await prisma.systemLicense.findFirst()
    
    if (!license) {
      console.log('❌ License bulunamadı!')
      console.log('💡 Kurulum için: npx tsx scripts/setup-license.ts iddaasohbet.com\n')
      process.exit(1)
    }
    
    // Domain'i decrypt et
    const domain = decryptDomain(license.encryptedDomain)
    
    // Hash doğrula - Sabit timestamp kullan
    const fixedTimestamp = 1730415600000 // 2024-11-01 fixed
    const isHashValid = verifyLicenseHash(
      domain,
      fixedTimestamp,
      license.licenseHash
    )
    
    console.log('📋 LICENSE BİLGİLERİ:\n')
    console.log(`   ID: ${license.id}`)
    console.log(`   Domain: ${domain}`)
    console.log(`   Durum: ${license.active ? '✅ Aktif' : '❌ Pasif'}`)
    console.log(`   Hash: ${isHashValid ? '✅ Geçerli' : '❌ Geçersiz'}`)
    console.log(`   Oluşturulma: ${new Date(license.createdAt).toLocaleString('tr-TR')}`)
    
    if (license.expiresAt) {
      const now = new Date()
      const expires = new Date(license.expiresAt)
      const daysLeft = Math.ceil((expires.getTime() - now.getTime()) / (1000 * 60 * 60 * 24))
      
      console.log(`   Bitiş Tarihi: ${expires.toLocaleString('tr-TR')}`)
      
      if (daysLeft > 0) {
        console.log(`   Kalan Süre: ${daysLeft} gün`)
      } else {
        console.log(`   ⚠️  SÜRESİ DOLMUŞ! (${Math.abs(daysLeft)} gün önce)`)
      }
    } else {
      console.log(`   Bitiş Tarihi: ♾️  Sınırsız`)
    }
    
    console.log('\n' + '='.repeat(50))
    
    if (license.active && isHashValid) {
      console.log('✅ LİSANS GEÇERLİ - Sistem çalışıyor!\n')
    } else {
      console.log('❌ LİSANS GEÇERSİZ - Sistem çalışmayacak!\n')
    }
    
  } catch (error) {
    console.error('\n❌ HATA:', error)
    process.exit(1)
  } finally {
    await prisma.$disconnect()
  }
}

checkLicense()

