import crypto from 'crypto'

// CRITICAL: Bu değerleri değiştirme! Sistem çalışmaz.
const ENCRYPTION_KEY = 'a8f5f167f44f4964e6c998dee827110c' // 32 chars
const ALGORITHM = 'aes-256-cbc'

/**
 * Domain'i şifreler
 * Müşteri database'de şifreli olarak saklanır
 */
export function encryptDomain(domain: string): string {
  const iv = crypto.randomBytes(16)
  const cipher = crypto.createCipheriv(ALGORITHM, Buffer.from(ENCRYPTION_KEY), iv)
  
  let encrypted = cipher.update(domain, 'utf8', 'hex')
  encrypted += cipher.final('hex')
  
  return iv.toString('hex') + ':' + encrypted
}

/**
 * Şifreli domain'i çözer
 */
export function decryptDomain(encryptedData: string): string {
  try {
    const parts = encryptedData.split(':')
    const iv = Buffer.from(parts[0], 'hex')
    const encrypted = parts[1]
    
    const decipher = crypto.createDecipheriv(ALGORITHM, Buffer.from(ENCRYPTION_KEY), iv)
    
    let decrypted = decipher.update(encrypted, 'hex', 'utf8')
    decrypted += decipher.final('utf8')
    
    return decrypted
  } catch {
    return ''
  }
}

/**
 * License key hash oluşturur
 */
export function createLicenseHash(domain: string, timestamp: number): string {
  const secret = 'IDDAASOHBET_LICENSE_2025_SECRET_KEY_NEVER_CHANGE'
  const data = `${domain}:${timestamp}:${secret}`
  return crypto.createHash('sha512').update(data).digest('hex')
}

/**
 * License key doğrular
 */
export function verifyLicenseHash(domain: string, timestamp: number, hash: string): boolean {
  const expectedHash = createLicenseHash(domain, timestamp)
  return expectedHash === hash
}

/**
 * Domain kontrolü - Runtime'da çalışır
 */
export function getCurrentDomain(): string {
  if (typeof window !== 'undefined') {
    return window.location.hostname
  }
  
  // Server-side
  return process.env.VERCEL_URL || process.env.NEXT_PUBLIC_DOMAIN || 'localhost'
}

/**
 * ANA LİSANS KONTROLÜ
 * Bu fonksiyon kritik noktalarda çağrılır
 */
export async function validateLicense(currentDomain?: string): Promise<boolean> {
  try {
    const domain = currentDomain || getCurrentDomain()
    
    // Localhost için development mode
    if (domain.includes('localhost') || domain.includes('127.0.0.1')) {
      return true
    }
    
    // Database'den lisans bilgisini çek
    const { prisma } = await import('@/lib/db')
    
    const license = await prisma.systemLicense.findFirst()
    
    if (!license) {
      console.error('❌ License not found in database')
      return false
    }
    
    // 1. Aktif mi kontrolü
    if (!license.active) {
      console.error('❌ License is not active')
      return false
    }
    
    // 2. Süre kontrolü
    if (license.expiresAt && new Date() > new Date(license.expiresAt)) {
      console.error('❌ License expired')
      return false
    }
    
    // 3. Şifreli domain kontrolü
    const decryptedDomain = decryptDomain(license.encryptedDomain)
    if (decryptedDomain !== domain && !domain.includes(decryptedDomain)) {
      console.error(`❌ Domain mismatch: ${domain} vs ${decryptedDomain}`)
      return false
    }
    
    // 4. Hash kontrolü - Sabit timestamp kullan
    const fixedTimestamp = 1730415600000 // 2024-11-01 fixed
    const isHashValid = verifyLicenseHash(
      decryptedDomain, 
      fixedTimestamp, 
      license.licenseHash
    )
    
    if (!isHashValid) {
      console.error('❌ License hash invalid')
      return false
    }
    
    // Tüm kontroller başarılı
    return true
    
  } catch (error) {
    console.error('❌ License validation error:', error)
    return false
  }
}

/**
 * License check wrapper - UI'da kullan
 */
export function useLicenseProtection() {
  if (typeof window === 'undefined') return
  
  validateLicense().then(valid => {
    if (!valid) {
      // Sayfayı kilitle
      document.body.innerHTML = `
        <div style="
          display: flex;
          align-items: center;
          justify-content: center;
          height: 100vh;
          background: #000;
          color: #ff4444;
          font-family: monospace;
          text-align: center;
          flex-direction: column;
          gap: 20px;
        ">
          <h1 style="font-size: 48px;">🔒 GEÇERSİZ LİSANS</h1>
          <p style="font-size: 20px; opacity: 0.8;">Bu yazılım yetkisiz kullanımda.</p>
          <p style="font-size: 16px; opacity: 0.6;">Lütfen yazılım sağlayıcınızla iletişime geçin.</p>
          <code style="background: #222; padding: 10px 20px; border-radius: 8px; margin-top: 20px;">
            License Error: Domain Verification Failed
          </code>
        </div>
      `
      throw new Error('Invalid License')
    }
  })
}

