'use client'

import Link from 'next/link'
import { useSession, signOut } from 'next-auth/react'
import { Button } from '@/components/ui/button'
import { Avatar, AvatarFallback, AvatarImage } from '@/components/ui/avatar'
import { Trophy, User, Search, Menu, Flame, LogOut, Settings, LayoutDashboard, X, Home, BarChart3, Users, Radio, MessageCircle, Bot, Sparkles } from 'lucide-react'
import DmUnreadBell from './DmUnreadBell'
import { useEffect, useState } from 'react'
import { usePathname } from 'next/navigation'
import LiveScoresTicker from './LiveScoresTicker'

export default function Header() {
  const { data: session, status } = useSession()
  const pathname = usePathname()
  const [showUserMenu, setShowUserMenu] = useState(false)
  const [showMobileMenu, setShowMobileMenu] = useState(false)

  const handleSignOut = async () => {
    await signOut({ callbackUrl: '/' })
  }
  
  useEffect(() => {
    if (showMobileMenu) {
      document?.body?.classList?.add('overflow-hidden')
    } else {
      document?.body?.classList?.remove('overflow-hidden')
    }
    return () => document?.body?.classList?.remove('overflow-hidden')
  }, [showMobileMenu])

  const navigationLinks = [
    { name: 'Ana Sayfa', href: '/', icon: Home },
    { name: 'Kuponlar', href: '/kuponlar', icon: Flame },
    { name: 'Canlı Sohbet', href: '/sohbet', icon: MessageCircle, badge: true },
    { name: 'Canlı Skorlar', href: '/canli-skorlar', icon: Radio, badge: true },
    { name: 'Tahmincilar', href: '/tahmincilar', icon: Users },
    { name: 'İstatistikler', href: '/istatistikler', icon: BarChart3 },
  ]

  return (
    <>
    <header className="w-full glass border-b border-white/5">
      <div className="container mx-auto flex h-16 items-center justify-between px-4">
        {/* Logo */}
        <Link href="/" className="flex items-center space-x-2 group">
          <div className="relative">
            <Trophy className="h-8 w-8 text-green-500 group-hover:text-yellow-400 transition-all duration-300" />
            <div className="absolute inset-0 bg-green-500 blur-xl opacity-20 group-hover:opacity-40 transition-opacity"></div>
          </div>
          <span className="text-2xl font-bold">
            <span className="gradient-text neon-text-green">İddaa</span>
            <span className="text-yellow-400 neon-text-yellow">Sohbet</span>
          </span>
        </Link>

        {/* Desktop Navigation */}
        <nav className="hidden md:flex items-center space-x-1">
          <Link href="/">
            <Button variant="ghost" className="text-foreground/80 hover:text-green-400 hover:bg-white/5 transition-all">
              Ana Sayfa
            </Button>
          </Link>
          <Link href="/kuponlar">
            <Button variant="ghost" className="text-foreground/80 hover:text-green-400 hover:bg-white/5 transition-all">
              <Flame className="h-4 w-4 mr-1 text-orange-500" />
              Kuponlar
            </Button>
          </Link>
          <Link href="/canli-skorlar">
            <Button variant="ghost" className="text-foreground/80 hover:text-green-400 hover:bg-white/5 transition-all relative">
              <Radio className="h-4 w-4 mr-1 text-red-500" />
              <span className="relative">
                Canlı Skorlar
                <span className="absolute -top-1 -right-1 flex h-2 w-2">
                  <span className="animate-ping absolute inline-flex h-full w-full rounded-full bg-red-400 opacity-75"></span>
                  <span className="relative inline-flex rounded-full h-2 w-2 bg-red-500"></span>
                </span>
              </span>
            </Button>
          </Link>
          <Link href="/sohbet">
            <Button variant="ghost" className="text-foreground/80 hover:text-green-400 hover:bg-white/5 transition-all relative">
              <Radio className="h-4 w-4 mr-1 text-red-500" />
              <span className="relative">
                Canlı Sohbet
                <span className="absolute -top-1 -right-1 flex h-2 w-2">
                  <span className="animate-ping absolute inline-flex h-full w-full rounded-full bg-red-400 opacity-75"></span>
                  <span className="relative inline-flex rounded-full h-2 w-2 bg-red-500"></span>
                </span>
              </span>
            </Button>
          </Link>
          <Link href="/tahmincilar">
            <Button variant="ghost" className="text-foreground/80 hover:text-green-400 hover:bg-white/5 transition-all">
              Tahmincilar
            </Button>
          </Link>
          <Link href="/istatistikler">
            <Button variant="ghost" className="text-foreground/80 hover:text-green-400 hover:bg-white/5 transition-all">
              İstatistikler
            </Button>
          </Link>
        </nav>

        {/* Right Side */}
        <div className="flex items-center space-x-3">
          <Button 
            variant="ghost" 
            size="icon" 
            className="hidden md:flex hover:bg-white/5 hover:text-green-400 transition-all"
          >
            <Search className="h-5 w-5" />
          </Button>
          
          <DmUnreadBell />
          
          {status === 'loading' ? (
            <div className="h-8 w-8 rounded-full animate-pulse bg-white/10"></div>
          ) : session ? (
            <div className="relative hidden md:block">
              <button
                onClick={() => setShowUserMenu(!showUserMenu)}
                className="flex items-center space-x-2 hover:bg-white/5 rounded-lg px-3 py-2 transition-all"
              >
                <Avatar className="h-8 w-8 border-2 border-green-500/50">
                  <AvatarImage src={session.user?.image || ''} alt={session.user?.name || ''} />
                  <AvatarFallback className="bg-gradient-to-br from-green-500 to-yellow-400 text-black font-bold">
                    {session.user?.name?.charAt(0).toUpperCase() || 'U'}
                  </AvatarFallback>
                </Avatar>
                <span className="text-sm font-medium">{session.user?.name}</span>
              </button>

              {showUserMenu && (
                <>
                  <div 
                    className="fixed inset-0 z-40" 
                    onClick={() => setShowUserMenu(false)}
                  ></div>
                  <div className="absolute right-0 top-12 w-56 glass-dark border border-white/10 rounded-lg shadow-2xl z-50 overflow-hidden">
                    <div className="p-4 border-b border-white/10">
                      <p className="text-sm font-semibold">{session.user?.name}</p>
                      <p className="text-xs text-foreground/60 truncate">{session.user?.email}</p>
                      <p className="text-xs text-green-400 mt-1">@{session.user?.username}</p>
                    </div>
                    <div className="py-2">
                      <Link href={`/profil/${session.user?.username}`}>
                        <button className="w-full flex items-center space-x-3 px-4 py-2 hover:bg-white/5 transition-all text-left">
                          <User className="h-4 w-4 text-foreground/70" />
                          <span className="text-sm">Profilim</span>
                        </button>
                      </Link>
                      <Link href="/hesap/ayarlar">
                        <button className="w-full flex items-center space-x-3 px-4 py-2 hover:bg-white/5 transition-all text-left">
                          <Settings className="h-4 w-4 text-foreground/70" />
                          <span className="text-sm">Ayarlar</span>
                        </button>
                      </Link>
                      {session.user?.role === 'ADMIN' && (
                        <Link href="/admin/dashboard">
                          <button className="w-full flex items-center space-x-3 px-4 py-2 hover:bg-white/5 transition-all text-left border-t border-white/10">
                            <LayoutDashboard className="h-4 w-4 text-yellow-400" />
                            <span className="text-sm text-yellow-400 font-semibold">Admin Panel</span>
                          </button>
                        </Link>
                      )}
                    </div>
                    <div className="border-t border-white/10">
                      <button 
                        onClick={handleSignOut}
                        className="w-full flex items-center space-x-3 px-4 py-3 hover:bg-red-500/10 transition-all text-left text-red-400"
                      >
                        <LogOut className="h-4 w-4" />
                        <span className="text-sm font-medium">Çıkış Yap</span>
                      </button>
                    </div>
                  </div>
                </>
              )}
            </div>
          ) : (
            <>
              <Link href="/giris" className="hidden md:block">
                <Button 
                  variant="outline" 
                  className="border-white/10 hover:border-green-500/50 hover:bg-green-500/10 hover:text-green-400 transition-all"
                >
                  Giriş Yap
                </Button>
              </Link>
              <Link href="/kayit" className="hidden md:flex">
                <Button className="bg-gradient-to-r from-green-500 to-yellow-400 hover:from-green-600 hover:to-yellow-500 text-black font-semibold btn-premium">
                  Kayıt Ol
                </Button>
              </Link>
            </>
          )}
          
          <Button 
            variant="ghost" 
            size="icon" 
            className="md:hidden hover:bg-white/5"
            onClick={() => setShowMobileMenu(!showMobileMenu)}
          >
            <div className="relative w-5 h-5">
              <span className={`absolute inset-0 transition-all duration-300 ${showMobileMenu ? 'opacity-0 rotate-90 scale-0' : 'opacity-100 rotate-0 scale-100'}`}>
                <Menu className="h-5 w-5" />
              </span>
              <span className={`absolute inset-0 transition-all duration-300 ${showMobileMenu ? 'opacity-100 rotate-0 scale-100' : 'opacity-0 -rotate-90 scale-0'}`}>
                <X className="h-5 w-5" />
              </span>
            </div>
          </Button>
        </div>
      </div>
    </header>
    
    <LiveScoresTicker />

      {/* Mobile Menu - PRO Drawer */}
      <div className={`fixed inset-y-0 right-0 w-[85vw] max-w-sm md:hidden z-[9999] glass-dark border-l border-green-500/30 shadow-2xl transform transition-transform duration-500 ease-out ${
        showMobileMenu ? 'translate-x-0' : 'translate-x-full'
      }`}>
        <div className="h-full flex flex-col relative overflow-hidden">
          
          {/* Gradient Overlay */}
          <div className="absolute inset-0 bg-gradient-to-br from-green-500/5 via-transparent to-yellow-500/5 pointer-events-none"></div>
          
          {/* Header - Simple Close Button */}
          <div className="relative px-4 py-4 flex items-center justify-between border-b border-white/5 backdrop-blur-sm">
            <div className="flex items-center gap-2">
              <Trophy className="h-6 w-6 text-green-500" />
              <span className="text-sm font-bold gradient-text">İddaa Sohbet</span>
            </div>
                <Button 
                  variant="ghost" 
                  size="icon"
                  onClick={() => setShowMobileMenu(false)}
                className="hover:bg-red-500/20 hover:text-red-400 transition-all duration-300 rounded-lg"
                >
                  <X className="h-5 w-5" />
                </Button>
          </div>

          {/* Navigation - PRO List */}
          <nav className="flex-1 px-4 pt-4 pb-4 space-y-1.5 relative overflow-y-auto">
            {navigationLinks.map((item, index) => {
              const isActive = pathname === item.href
              return (
                <Link 
                  key={item.name} 
                  href={item.href} 
                  onClick={() => setShowMobileMenu(false)}
                  className="block"
                >
                  <div className={`group relative overflow-hidden rounded-lg transition-all duration-300 ${
                    isActive ? 'scale-[1.02]' : 'hover:scale-[1.02]'
                  }`}>
                    {isActive && (
                      <div className="absolute inset-0 bg-gradient-to-r from-green-500/20 to-yellow-500/20 blur-xl"></div>
                    )}
                    <div className={`relative px-4 py-3.5 border rounded-lg transition-all duration-300 backdrop-blur-sm ${
                      isActive
                        ? 'border-green-500/50 bg-gradient-to-r from-green-500/20 to-yellow-500/10 text-green-400 shadow-lg shadow-green-500/20'
                        : 'border-white/5 bg-white/[0.02] group-hover:bg-white/[0.05] group-hover:border-green-500/30 text-white/90 group-hover:text-white'
                    }`}>
                      <span className="font-semibold text-sm inline-flex items-center gap-3">
                        <item.icon className={`h-5 w-5 ${isActive ? 'text-green-400' : ''}`} />
                        {item.name}
                        {item.badge && (
                          <span className="flex h-2 w-2 ml-auto">
                            <span className="animate-ping absolute inline-flex h-2 w-2 rounded-full bg-red-400 opacity-75"></span>
                            <span className="relative inline-flex rounded-full h-2 w-2 bg-red-500"></span>
                          </span>
                        )}
                      </span>
                </div>
                </div>
              </Link>
              )
            })}

              {session && (
                <>
                <div className="my-3 h-px bg-gradient-to-r from-transparent via-white/10 to-transparent"></div>
                  
                  <Link href={`/profil/${session.user?.username}`} onClick={() => setShowMobileMenu(false)} className="block">
                    <div className="group relative overflow-hidden rounded-lg transition-all duration-300 hover:scale-[1.02]">
                      <div className="relative px-4 py-3.5 border border-white/5 rounded-lg bg-white/[0.02] group-hover:bg-white/[0.05] group-hover:border-green-500/30 transition-all duration-300 backdrop-blur-sm">
                        <span className="text-white/90 font-semibold text-sm group-hover:text-white transition-all duration-300 inline-flex items-center gap-3">
                          <User className="h-5 w-5" />
                          Profilim
                        </span>
                      </div>
                    </div>
                  </Link>

                  <Link href="/hesap/ayarlar" onClick={() => setShowMobileMenu(false)} className="block">
                    <div className="group relative overflow-hidden rounded-lg transition-all duration-300 hover:scale-[1.02]">
                      <div className="relative px-4 py-3.5 border border-white/5 rounded-lg bg-white/[0.02] group-hover:bg-white/[0.05] group-hover:border-green-500/30 transition-all duration-300 backdrop-blur-sm">
                        <span className="text-white/90 font-semibold text-sm group-hover:text-white transition-all duration-300 inline-flex items-center gap-3">
                          <Settings className="h-5 w-5" />
                          Ayarlar
                        </span>
                      </div>
                    </div>
                  </Link>

                  {session.user?.role === 'ADMIN' && (
                    <Link href="/admin/dashboard" onClick={() => setShowMobileMenu(false)} className="block">
                      <div className="group relative overflow-hidden rounded-lg transition-all duration-300 hover:scale-[1.02]">
                        <div className="absolute inset-0 bg-gradient-to-r from-yellow-500/10 to-orange-500/10 blur-xl opacity-0 group-hover:opacity-100 transition-opacity"></div>
                        <div className="relative px-4 py-3.5 border border-yellow-500/30 rounded-lg bg-gradient-to-r from-yellow-500/10 to-orange-500/5 group-hover:from-yellow-500/20 group-hover:to-orange-500/10 group-hover:border-yellow-500/50 transition-all duration-300 backdrop-blur-sm shadow-lg shadow-yellow-500/10">
                          <span className="text-yellow-400 font-semibold text-sm group-hover:text-yellow-300 transition-all duration-300 inline-flex items-center gap-3">
                            <LayoutDashboard className="h-5 w-5" />
                            Admin Panel
                          </span>
                        </div>
                      </div>
                    </Link>
                  )}
                </>
              )}
            </nav>

          {/* Footer Actions - PRO Buttons */}
          <div className="relative p-4 border-t border-white/5 backdrop-blur-sm">
              {!session ? (
                <div className="flex gap-3">
                  <Link href="/giris" onClick={() => setShowMobileMenu(false)} className="flex-1">
                    <Button variant="outline" className="w-full h-12 text-sm font-semibold border-white/10 hover:border-green-500/50 hover:bg-green-500/10 hover:text-green-400 transition-all rounded-lg backdrop-blur-sm">
                      Giriş Yap
                    </Button>
                  </Link>
                  <Link href="/kayit" onClick={() => setShowMobileMenu(false)} className="flex-1">
                    <Button className="w-full h-12 text-sm font-bold bg-gradient-to-r from-green-500 to-yellow-400 hover:from-green-600 hover:to-yellow-500 text-black transition-all rounded-lg shadow-lg shadow-green-500/20 hover:shadow-green-500/40">
                      Kayıt Ol
                    </Button>
                  </Link>
                </div>
              ) : (
                  <Button 
                    onClick={() => {
                      setShowMobileMenu(false)
                      handleSignOut()
                    }}
                    variant="outline"
                    className="w-full h-12 text-sm font-semibold border-red-500/30 text-red-400 hover:bg-red-500/20 hover:border-red-500/50 transition-all rounded-lg backdrop-blur-sm"
                  >
                    <LogOut className="h-5 w-5 mr-2" />
                    Çıkış Yap
                  </Button>
              )}
            </div>
          </div>
      </div>

      {/* Backdrop - PRO */}
      {showMobileMenu && (
        <div 
          className="fixed inset-0 bg-black/60 backdrop-blur-sm z-[9998] md:hidden transition-opacity duration-500"
          onClick={() => setShowMobileMenu(false)}
        />
      )}
    </>
  )
}
