'use client'

import { useEffect, useState } from 'react'
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card'
import { Badge } from '@/components/ui/badge'
import { Button } from '@/components/ui/button'
import { Input } from '@/components/ui/input'
import { 
  Search,
  SlidersHorizontal,
  MessageSquare,
  Eye,
  Edit,
  Trash2,
  CheckCircle,
  XCircle,
  Clock,
  AlertTriangle,
  Flag,
  ThumbsUp,
  ThumbsDown,
  MoreVertical,
  Download,
  TrendingUp,
  User,
  FileText,
  Calendar,
  X,
  AlertCircle,
  Activity
} from 'lucide-react'

export default function AdminYorumlarPage() {
  const [searchQuery, setSearchQuery] = useState('')
  const [selectedComments, setSelectedComments] = useState<string[]>([])
  const [loading, setLoading] = useState(true)
  const [page, setPage] = useState(1)
  const [limit] = useState(20)
  
  // Modal states
  const [showEditModal, setShowEditModal] = useState(false)
  const [showDeleteModal, setShowDeleteModal] = useState(false)
  const [selectedComment, setSelectedComment] = useState<any>(null)

  // Real data states
  const [stats, setStats] = useState([
    { label: 'Toplam Yorum', value: '0', icon: MessageSquare, color: 'blue' }
  ])

  const [comments, setComments] = useState<any[]>([])
  const [total, setTotal] = useState(0)

  // Load comments
  const loadComments = async () => {
    setLoading(true)
    try {
      const params = new URLSearchParams()
      params.set('page', String(page))
      params.set('limit', String(limit))
      if (searchQuery) params.set('search', searchQuery)

      const res = await fetch(`/api/admin/comments?${params.toString()}`, { cache: 'no-store' })
      const data = await res.json()
      
      if (data.comments) {
        setComments(data.comments)
        setTotal(data.pagination?.total || 0)
        
        // Update stats
        if (data.stats) {
          setStats([
            { label: 'Toplam Yorum', value: String(data.stats.total || 0), icon: MessageSquare, color: 'blue' }
          ])
        }
      }
    } catch (error) {
      console.error('Error loading comments:', error)
    } finally {
      setLoading(false)
    }
  }

  useEffect(() => {
    loadComments()
  }, [page])

  // Search with debounce
  useEffect(() => {
    const timer = setTimeout(() => {
      if (page === 1) {
        loadComments()
      } else {
        setPage(1)
      }
    }, 500)
    return () => clearTimeout(timer)
  }, [searchQuery])

  const handleSelectComment = (id: string) => {
    setSelectedComments(prev => 
      prev.includes(id) ? prev.filter(i => i !== id) : [...prev, id]
    )
  }

  const handleSelectAll = () => {
    if (selectedComments.length === comments.length) {
      setSelectedComments([])
    } else {
      setSelectedComments(comments.map(c => c.id))
    }
  }

  // Delete Comment
  const handleDeleteComment = async () => {
    try {
      const res = await fetch(`/api/admin/comments?commentId=${selectedComment.id}`, {
        method: 'DELETE'
      })

      const data = await res.json()
      if (data.success) {
        alert('✅ Yorum silindi!')
        setShowDeleteModal(false)
        setSelectedComment(null)
        await loadComments()
      } else {
        alert('❌ Hata: ' + (data.error || 'Yorum silinemedi'))
      }
    } catch (error) {
      console.error('Error deleting comment:', error)
      alert('❌ Yorum silinemedi')
    }
  }

  // Bulk Delete
  const handleBulkDelete = async () => {
    if (!confirm(`${selectedComments.length} yorumu silmek istediğinizden emin misiniz?`)) return
    
    try {
      await Promise.all(
        selectedComments.map(commentId =>
          fetch(`/api/admin/comments?commentId=${commentId}`, { method: 'DELETE' })
        )
      )
      alert('✅ Seçili yorumlar silindi!')
      setSelectedComments([])
      await loadComments()
    } catch (error) {
      console.error('Error bulk deleting:', error)
      alert('❌ Yorumlar silinemedi')
    }
  }

  const openDeleteModal = (comment: any) => {
    setSelectedComment(comment)
    setShowDeleteModal(true)
  }

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleString('tr-TR')
  }

  const getRelativeTime = (dateString: string) => {
    const now = new Date()
    const date = new Date(dateString)
    const diffMs = now.getTime() - date.getTime()
    const diffMins = Math.floor(diffMs / 60000)
    
    if (diffMins < 1) return 'Az önce'
    if (diffMins < 60) return `${diffMins} dk önce`
    
    const diffHours = Math.floor(diffMins / 60)
    if (diffHours < 24) return `${diffHours} saat önce`
    
    const diffDays = Math.floor(diffHours / 24)
    if (diffDays < 30) return `${diffDays} gün önce`
    
    return formatDate(dateString)
  }

  return (
    <div className="space-y-6">
      {/* Page Header */}
      <div className="flex flex-col md:flex-row justify-between items-start md:items-center gap-4">
        <div>
          <h1 className="text-3xl font-bold gradient-text mb-2">Yorum Yönetimi</h1>
          <p className="text-foreground/60">Tüm yorumları görüntüle, düzenle ve yönet</p>
        </div>
        <div className="flex items-center space-x-3">
          <Button variant="outline" className="glass border-white/10" onClick={() => loadComments()}>
            <Activity className="h-4 w-4 mr-2" />
            Yenile
          </Button>
        </div>
      </div>

      {/* Stats Cards */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
        {stats.map((stat, index) => {
          const Icon = stat.icon
          const colorClasses = {
            blue: { bg: 'bg-blue-500/10 border-blue-500/30', text: 'text-blue-400' }
          }
          const colors = colorClasses[stat.color as keyof typeof colorClasses] || colorClasses.blue
          return (
            <Card key={index} className="glass-dark border-white/5 hover:border-white/10 transition-all group cursor-pointer">
              <CardContent className="p-6">
                <div className="flex items-center justify-between mb-4">
                  <div className={`h-12 w-12 rounded-xl ${colors.bg} border flex items-center justify-center group-hover:scale-110 transition-transform`}>
                    <Icon className={`h-6 w-6 ${colors.text}`} />
                  </div>
                </div>
                <p className="text-foreground/60 text-sm mb-1">{stat.label}</p>
                <p className="text-3xl font-bold">{stat.value}</p>
              </CardContent>
            </Card>
          )
        })}
      </div>

      {/* Filters & Search */}
      <Card className="glass-dark border-white/5">
        <CardContent className="p-6">
          <div className="flex flex-col md:flex-row gap-4">
            <div className="flex-1 relative">
              <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-foreground/40" />
              <Input
                type="search"
                placeholder="Kullanıcı, kupon veya yorum içeriği ara..."
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                className="glass-dark border-white/10 pl-10"
              />
            </div>

          </div>

          {selectedComments.length > 0 && (
            <div className="mt-4 pt-4 border-t border-white/5 flex items-center justify-between">
              <p className="text-sm text-foreground/60">
                <strong className="text-orange-400">{selectedComments.length}</strong> yorum seçildi
              </p>
              <div className="flex items-center space-x-2">
                <Button size="sm" variant="outline" className="glass border-red-500/30 text-red-400 hover:bg-red-500/10" onClick={handleBulkDelete}>
                  <Trash2 className="h-4 w-4 mr-1" />
                  Toplu Sil
                </Button>
              </div>
            </div>
          )}
        </CardContent>
      </Card>

      {/* Comments List */}
      <Card className="glass-dark border-white/5">
        <CardHeader className="border-b border-white/5">
          <CardTitle className="flex items-center justify-between">
            <span className="flex items-center space-x-2">
              <MessageSquare className="h-5 w-5 text-blue-400" />
              <span>Yorum Listesi ({comments.length})</span>
            </span>
            <Button variant="ghost" size="sm" onClick={handleSelectAll} className="text-xs">
              {selectedComments.length === comments.length ? 'Seçimi Kaldır' : 'Tümünü Seç'}
            </Button>
          </CardTitle>
        </CardHeader>
        <CardContent className="p-6">
          {loading ? (
            <div className="p-12 text-center">
              <div className="inline-block h-8 w-8 animate-spin rounded-full border-4 border-solid border-orange-500 border-r-transparent"></div>
              <p className="mt-4 text-foreground/60">Yorumlar yükleniyor...</p>
            </div>
          ) : comments.length === 0 ? (
            <div className="p-12 text-center">
              <MessageSquare className="h-16 w-16 mx-auto text-foreground/20 mb-4" />
              <p className="text-foreground/60">Henüz yorum bulunmuyor</p>
            </div>
          ) : (
            <div className="space-y-4">
              {comments.map((comment) => (
                <div 
                  key={comment.id}
                  className={`glass-dark p-5 rounded-lg border transition-all hover:border-white/10 ${
                    selectedComments.includes(comment.id) ? 'border-orange-500/50 bg-orange-500/5' : 'border-white/5'
                  }`}
                >
                  <div className="flex items-start space-x-4">
                    <input
                      type="checkbox"
                      checked={selectedComments.includes(comment.id)}
                      onChange={() => handleSelectComment(comment.id)}
                      className="rounded border-white/20 bg-white/5 cursor-pointer mt-1"
                    />

                    <div className="h-12 w-12 rounded-full bg-gradient-to-br from-blue-500 to-purple-500 flex items-center justify-center text-white font-bold text-lg flex-shrink-0">
                      {(comment.user?.name || comment.user?.username || 'U').charAt(0).toUpperCase()}
                    </div>

                    <div className="flex-1 min-w-0">
                      <div className="flex flex-col md:flex-row md:items-center justify-between gap-2 mb-3">
                        <div className="flex flex-col md:flex-row md:items-center gap-2">
                          <div>
                            <h4 className="font-semibold text-sm">{comment.user?.name || 'Anonim'}</h4>
                            <p className="text-xs text-foreground/60">@{comment.user?.username || 'user'}</p>
                          </div>
                        </div>
                        <span className="text-xs text-foreground/60">{getRelativeTime(comment.createdAt)}</span>
                      </div>

                      <div className="flex items-center space-x-2 mb-3 text-xs text-foreground/60">
                        <FileText className="h-3 w-3" />
                        <span>Kupon:</span>
                        <a href={`/kuponlar/${comment.couponId}`} className="text-blue-400 hover:underline">
                          {comment.coupon?.title || '#' + comment.couponId}
                        </a>
                      </div>

                      <p className="text-sm mb-4 text-foreground/80">
                        {comment.content}
                      </p>

                      <div className="flex items-center justify-between">
                        <div className="flex items-center space-x-4 text-xs text-foreground/60">
                          <div className="flex items-center space-x-1">
                            <Calendar className="h-3 w-3" />
                            <span>{formatDate(comment.createdAt)}</span>
                          </div>
                        </div>

                        <div className="flex items-center space-x-2">
                          <Button size="sm" variant="ghost" className="h-8 w-8 p-0 hover:bg-red-500/10 hover:text-red-400" onClick={() => openDeleteModal(comment)}>
                            <Trash2 className="h-4 w-4" />
                          </Button>
                        </div>
                      </div>
                    </div>
                  </div>
                </div>
              ))}
            </div>
          )}

          <div className="mt-6 pt-6 border-t border-white/5">
            <p className="text-sm text-foreground/60">
              Toplam <strong className="text-orange-400">{total}</strong> yorumdan <strong className="text-blue-400">{comments.length}</strong> tanesi gösteriliyor
            </p>
          </div>
        </CardContent>
      </Card>

      {/* Delete Modal */}
      {showDeleteModal && selectedComment && (
        <div className="fixed inset-0 bg-black/80 backdrop-blur-sm flex items-center justify-center z-50 p-4">
          <Card className="glass-dark border-red-500/30 w-full max-w-md">
            <CardHeader className="border-b border-white/5">
              <CardTitle className="flex items-center justify-between">
                <span className="flex items-center space-x-2 text-red-400">
                  <AlertCircle className="h-5 w-5" />
                  <span>Yorum Sil</span>
                </span>
                <Button variant="ghost" size="sm" onClick={() => setShowDeleteModal(false)}>
                  <X className="h-5 w-5" />
                </Button>
              </CardTitle>
            </CardHeader>
            <CardContent className="p-6">
              <div className="glass p-4 rounded-lg mb-4">
                <p className="text-sm text-foreground/70">{selectedComment.content}</p>
              </div>
              <p className="text-sm text-red-400 mb-6">
                Bu yorumu silmek istediğinizden emin misiniz? Bu işlem geri alınamaz!
              </p>
              <div className="flex space-x-3">
                <Button 
                  onClick={handleDeleteComment}
                  className="flex-1 bg-gradient-to-r from-red-500 to-red-600 hover:from-red-600 hover:to-red-700"
                >
                  <Trash2 className="h-4 w-4 mr-2" />
                  Evet, Sil
                </Button>
                <Button 
                  variant="outline" 
                  className="flex-1 glass border-white/10" 
                  onClick={() => setShowDeleteModal(false)}
                >
                  İptal
                </Button>
              </div>
            </CardContent>
          </Card>
        </div>
      )}
    </div>
  )
}