'use client'

import { useEffect, useState } from 'react'
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card'
import { Badge } from '@/components/ui/badge'
import { Button } from '@/components/ui/button'
import { Input } from '@/components/ui/input'
import { 
  Search,
  SlidersHorizontal,
  Users,
  Eye,
  Edit,
  Trash2,
  Ban,
  CheckCircle,
  XCircle,
  UserPlus,
  TrendingUp,
  Mail,
  Calendar,
  MoreVertical,
  Download,
  Shield,
  Crown,
  Star,
  Activity,
  X,
  AlertCircle,
  Plus
} from 'lucide-react'

export default function AdminKullanicilarPage() {
  const [searchQuery, setSearchQuery] = useState('')
  const [filterRole, setFilterRole] = useState('all')
  const [selectedUsers, setSelectedUsers] = useState<string[]>([])
  const [loading, setLoading] = useState(true)
  const [page, setPage] = useState(1)
  const [limit] = useState(20)
  
  // Modal states
  const [showAddModal, setShowAddModal] = useState(false)
  const [showEditModal, setShowEditModal] = useState(false)
  const [showDeleteModal, setShowDeleteModal] = useState(false)
  const [showDetailModal, setShowDetailModal] = useState(false)
  const [selectedUser, setSelectedUser] = useState<any>(null)

  // Form state
  const [formData, setFormData] = useState({
    name: '',
    username: '',
    email: '',
    role: 'USER',
    verified: false
  })

  // Real data states
  const [stats, setStats] = useState([
    { label: 'Toplam Kullanıcı', value: '0', icon: Users, color: 'blue', trend: '+0%' },
    { label: 'Yeni Üyeler (30 gün)', value: '0', icon: UserPlus, color: 'green', trend: '+0%' },
    { label: 'Admin Kullanıcılar', value: '0', icon: Shield, color: 'red', trend: '+0%' },
    { label: 'Moderatörler', value: '0', icon: Star, color: 'purple', trend: '+0%' }
  ])

  const [users, setUsers] = useState<any[]>([])
  const [total, setTotal] = useState(0)

  // Load users
  const loadUsers = async () => {
    setLoading(true)
    try {
      const params = new URLSearchParams()
      params.set('page', String(page))
      params.set('limit', String(limit))
      if (filterRole !== 'all') params.set('role', filterRole)
      if (searchQuery) params.set('search', searchQuery)

      const res = await fetch(`/api/admin/users?${params.toString()}`, { cache: 'no-store' })
      const data = await res.json()
      
      if (data.users) {
        setUsers(data.users)
        setTotal(data.pagination?.total || 0)
        
        // Update stats
        if (data.stats) {
          setStats([
            { label: 'Toplam Kullanıcı', value: String(data.stats.total || 0), icon: Users, color: 'blue', trend: '+0%' },
            { label: 'Yeni Üyeler (30 gün)', value: String(data.stats.newUsers || 0), icon: UserPlus, color: 'green', trend: '+0%' },
            { label: 'Admin Kullanıcılar', value: String(data.stats.admins || 0), icon: Shield, color: 'red', trend: '+0%' },
            { label: 'Moderatörler', value: String(data.stats.moderators || 0), icon: Star, color: 'purple', trend: '+0%' }
          ])
        }
      }
    } catch (error) {
      console.error('Error loading users:', error)
    } finally {
      setLoading(false)
    }
  }

  useEffect(() => {
    loadUsers()
  }, [page, filterRole])

  // Search with debounce
  useEffect(() => {
    const timer = setTimeout(() => {
      if (page === 1) {
        loadUsers()
      } else {
        setPage(1)
      }
    }, 500)
    return () => clearTimeout(timer)
  }, [searchQuery])

  const getRoleBadge = (role: string) => {
    const r = role?.toUpperCase()
    switch (r) {
      case 'ADMIN':
        return <Badge className="bg-red-500/20 border-red-500/30 text-red-400"><Shield className="h-3 w-3 mr-1" />Admin</Badge>
      case 'MODERATOR':
        return <Badge className="bg-purple-500/20 border-purple-500/30 text-purple-400"><Star className="h-3 w-3 mr-1" />Moderatör</Badge>
      case 'USER':
      default:
        return <Badge className="bg-blue-500/20 border-blue-500/30 text-blue-400">Kullanıcı</Badge>
    }
  }

  const getStatusBadge = (verified: boolean) => {
    if (verified) {
      return <Badge className="bg-green-500/20 border-green-500/30 text-green-400"><CheckCircle className="h-3 w-3 mr-1" />Doğrulanmış</Badge>
    }
    return <Badge className="bg-gray-500/20 border-gray-500/30 text-gray-400">Doğrulanmamış</Badge>
  }

  const handleSelectUser = (id: string) => {
    setSelectedUsers(prev => 
      prev.includes(id) ? prev.filter(i => i !== id) : [...prev, id]
    )
  }

  const handleSelectAll = () => {
    if (selectedUsers.length === users.length) {
      setSelectedUsers([])
    } else {
      setSelectedUsers(users.map(u => u.id))
    }
  }

  // Edit User
  const handleEditUser = async () => {
    try {
      const res = await fetch('/api/admin/users', {
        method: 'PUT',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          userId: selectedUser.id,
          name: formData.name,
          username: formData.username,
          email: formData.email,
          role: formData.role,
          verified: formData.verified
        })
      })

      const data = await res.json()
      if (data.success) {
        alert('✅ Kullanıcı güncellendi!')
        setShowEditModal(false)
        setSelectedUser(null)
        resetForm()
        await loadUsers()
      } else {
        alert('❌ Hata: ' + (data.error || 'Kullanıcı güncellenemedi'))
      }
    } catch (error) {
      console.error('Error updating user:', error)
      alert('❌ Kullanıcı güncellenemedi')
    }
  }

  // Delete User
  const handleDeleteUser = async () => {
    try {
      const res = await fetch(`/api/admin/users?userId=${selectedUser.id}`, {
        method: 'DELETE'
      })

      const data = await res.json()
      if (data.success) {
        alert('✅ Kullanıcı silindi!')
        setShowDeleteModal(false)
        setSelectedUser(null)
        await loadUsers()
      } else {
        alert('❌ Hata: ' + (data.error || 'Kullanıcı silinemedi'))
      }
    } catch (error) {
      console.error('Error deleting user:', error)
      alert('❌ Kullanıcı silinemedi')
    }
  }

  // Bulk operations
  const handleBulkDelete = async () => {
    if (!confirm(`${selectedUsers.length} kullanıcıyı silmek istediğinizden emin misiniz?`)) return
    
    try {
      await Promise.all(
        selectedUsers.map(userId =>
          fetch(`/api/admin/users?userId=${userId}`, { method: 'DELETE' })
        )
      )
      alert('✅ Seçili kullanıcılar silindi!')
      setSelectedUsers([])
      await loadUsers()
    } catch (error) {
      console.error('Error bulk deleting:', error)
      alert('❌ Kullanıcılar silinemedi')
    }
  }

  const handleBulkVerify = async () => {
    try {
      await Promise.all(
        selectedUsers.map(userId =>
          fetch('/api/admin/users', {
            method: 'PUT',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ userId, verified: true })
          })
        )
      )
      alert('✅ Seçili kullanıcılar doğrulandı!')
      setSelectedUsers([])
      await loadUsers()
    } catch (error) {
      console.error('Error bulk verifying:', error)
      alert('❌ Kullanıcılar doğrulanamadı')
    }
  }

  const openEditModal = (user: any) => {
    setSelectedUser(user)
    setFormData({
      name: user.name || '',
      username: user.username || '',
      email: user.email || '',
      role: user.role || 'USER',
      verified: user.verified || false
    })
    setShowEditModal(true)
  }

  const openDeleteModal = (user: any) => {
    setSelectedUser(user)
    setShowDeleteModal(true)
  }

  const openDetailModal = (user: any) => {
    setSelectedUser(user)
    setShowDetailModal(true)
  }

  const resetForm = () => {
    setFormData({
      name: '',
      username: '',
      email: '',
      role: 'USER',
      verified: false
    })
  }

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('tr-TR')
  }

  const getRelativeTime = (dateString: string) => {
    const now = new Date()
    const date = new Date(dateString)
    const diffMs = now.getTime() - date.getTime()
    const diffMins = Math.floor(diffMs / 60000)
    
    if (diffMins < 1) return 'Az önce'
    if (diffMins < 60) return `${diffMins} dk önce`
    
    const diffHours = Math.floor(diffMins / 60)
    if (diffHours < 24) return `${diffHours} saat önce`
    
    const diffDays = Math.floor(diffHours / 24)
    if (diffDays < 30) return `${diffDays} gün önce`
    
    return formatDate(dateString)
  }

  return (
    <div className="space-y-6">
      {/* Page Header */}
      <div className="flex flex-col md:flex-row justify-between items-start md:items-center gap-4">
        <div>
          <h1 className="text-3xl font-bold gradient-text mb-2">Kullanıcı Yönetimi</h1>
          <p className="text-foreground/60">Tüm kullanıcıları görüntüle, düzenle ve yönet</p>
        </div>
        <div className="flex items-center space-x-3">
          <Button variant="outline" className="glass border-white/10" onClick={() => loadUsers()}>
            <Activity className="h-4 w-4 mr-2" />
            Yenile
          </Button>
        </div>
      </div>

      {/* Stats Cards */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
        {stats.map((stat, index) => {
          const Icon = stat.icon
          const colorClasses = {
            blue: { bg: 'bg-blue-500/10 border-blue-500/30', text: 'text-blue-400' },
            green: { bg: 'bg-green-500/10 border-green-500/30', text: 'text-green-400' },
            red: { bg: 'bg-red-500/10 border-red-500/30', text: 'text-red-400' },
            purple: { bg: 'bg-purple-500/10 border-purple-500/30', text: 'text-purple-400' }
          }
          const colors = colorClasses[stat.color as keyof typeof colorClasses] || colorClasses.blue
          return (
            <Card key={index} className="glass-dark border-white/5 hover:border-white/10 transition-all group cursor-pointer">
              <CardContent className="p-6">
                <div className="flex items-center justify-between mb-4">
                  <div className={`h-12 w-12 rounded-xl ${colors.bg} border flex items-center justify-center group-hover:scale-110 transition-transform`}>
                    <Icon className={`h-6 w-6 ${colors.text}`} />
                  </div>
                </div>
                <p className="text-foreground/60 text-sm mb-1">{stat.label}</p>
                <p className="text-3xl font-bold">{stat.value}</p>
              </CardContent>
            </Card>
          )
        })}
      </div>

      {/* Filters & Search */}
      <Card className="glass-dark border-white/5">
        <CardContent className="p-6">
          <div className="flex flex-col md:flex-row gap-4">
            <div className="flex-1 relative">
              <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-foreground/40" />
              <Input
                type="search"
                placeholder="İsim, email veya kullanıcı adı ara..."
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                className="glass-dark border-white/10 pl-10"
              />
            </div>

            <div className="flex items-center space-x-2">
              <SlidersHorizontal className="h-4 w-4 text-foreground/60" />
              <select
                value={filterRole}
                onChange={(e) => setFilterRole(e.target.value)}
                className="glass-dark border border-white/10 rounded-lg px-4 py-2 text-sm focus:outline-none focus:border-orange-500/50 min-w-[150px]"
              >
                <option value="all">Tüm Roller</option>
                <option value="ADMIN">Admin</option>
                <option value="MODERATOR">Moderatör</option>
                <option value="USER">Kullanıcı</option>
              </select>
            </div>
          </div>

          {selectedUsers.length > 0 && (
            <div className="mt-4 pt-4 border-t border-white/5 flex items-center justify-between">
              <p className="text-sm text-foreground/60">
                <strong className="text-orange-400">{selectedUsers.length}</strong> kullanıcı seçildi
              </p>
              <div className="flex items-center space-x-2">
                <Button 
                  size="sm" 
                  variant="outline" 
                  className="glass border-green-500/30 text-green-400 hover:bg-green-500/10"
                  onClick={handleBulkVerify}
                >
                  <CheckCircle className="h-4 w-4 mr-1" />
                  Doğrula
                </Button>
                <Button 
                  size="sm" 
                  variant="outline" 
                  className="glass border-red-500/30 text-red-400 hover:bg-red-500/10"
                  onClick={handleBulkDelete}
                >
                  <Trash2 className="h-4 w-4 mr-1" />
                  Sil
                </Button>
              </div>
            </div>
          )}
        </CardContent>
      </Card>

      {/* Users Table */}
      <Card className="glass-dark border-white/5">
        <CardHeader className="border-b border-white/5">
          <CardTitle className="flex items-center justify-between">
            <span className="flex items-center space-x-2">
              <Users className="h-5 w-5 text-blue-400" />
              <span>Kullanıcı Listesi ({users.length})</span>
            </span>
            <Button variant="ghost" size="sm" onClick={handleSelectAll} className="text-xs">
              {selectedUsers.length === users.length ? 'Seçimi Kaldır' : 'Tümünü Seç'}
            </Button>
          </CardTitle>
        </CardHeader>
        <CardContent className="p-0">
          <div className="overflow-x-auto">
            <table className="w-full">
              <thead className="border-b border-white/5 bg-white/5">
                <tr>
                  <th className="text-left p-4 text-sm font-semibold text-foreground/70">
                    <input
                      type="checkbox"
                      checked={selectedUsers.length === users.length}
                      onChange={handleSelectAll}
                      className="rounded border-white/20 bg-white/5 cursor-pointer"
                    />
                  </th>
                  <th className="text-left p-4 text-sm font-semibold text-foreground/70">Kullanıcı</th>
                  <th className="text-left p-4 text-sm font-semibold text-foreground/70">Email</th>
                  <th className="text-left p-4 text-sm font-semibold text-foreground/70">Rol</th>
                  <th className="text-left p-4 text-sm font-semibold text-foreground/70">Doğrulama</th>
                  <th className="text-left p-4 text-sm font-semibold text-foreground/70">İstatistikler</th>
                  <th className="text-left p-4 text-sm font-semibold text-foreground/70">Kayıt Tarihi</th>
                  <th className="text-right p-4 text-sm font-semibold text-foreground/70">İşlemler</th>
                </tr>
              </thead>
              <tbody>
                {users.map((user) => (
                  <tr 
                    key={user.id}
                    className="border-b border-white/5 hover:bg-white/5 transition-colors"
                  >
                    <td className="p-4">
                      <input
                        type="checkbox"
                        checked={selectedUsers.includes(user.id)}
                        onChange={() => handleSelectUser(user.id)}
                        className="rounded border-white/20 bg-white/5 cursor-pointer"
                      />
                    </td>
                    <td className="p-4">
                      <div className="flex items-center space-x-3">
                        <div className="relative">
                          <div className="h-10 w-10 rounded-full bg-gradient-to-br from-blue-500 to-purple-500 flex items-center justify-center text-white font-bold text-sm">
                            {(user.name || user.username || 'U').charAt(0).toUpperCase()}
                          </div>
                          {user.verified && (
                            <div className="absolute -bottom-1 -right-1 h-4 w-4 bg-green-500 rounded-full border-2 border-slate-900 flex items-center justify-center">
                              <CheckCircle className="h-2.5 w-2.5 text-white" />
                            </div>
                          )}
                        </div>
                        <div>
                          <p className="font-semibold text-sm">{user.name || 'İsimsiz'}</p>
                          <p className="text-xs text-foreground/60">@{user.username}</p>
                        </div>
                      </div>
                    </td>
                    <td className="p-4">
                      <div className="flex items-center space-x-2">
                        <Mail className="h-4 w-4 text-foreground/40" />
                        <span className="text-sm text-foreground/70">{user.email}</span>
                      </div>
                    </td>
                    <td className="p-4">
                      {getRoleBadge(user.role)}
                    </td>
                    <td className="p-4">
                      {getStatusBadge(user.verified)}
                    </td>
                    <td className="p-4">
                      <div className="flex flex-col space-y-1">
                        <span className="text-sm font-semibold text-blue-400">{user._count?.coupons || 0} Kupon</span>
                        <span className="text-xs text-green-400">{user.winRate?.toFixed(1) || '0.0'}% Başarı</span>
                      </div>
                    </td>
                    <td className="p-4">
                      <div className="flex flex-col space-y-1">
                        <div className="flex items-center space-x-2">
                          <Calendar className="h-4 w-4 text-foreground/40" />
                          <span className="text-xs text-foreground/60">{formatDate(user.createdAt)}</span>
                        </div>
                        <span className="text-xs text-foreground/40">{getRelativeTime(user.updatedAt)}</span>
                      </div>
                    </td>
                    <td className="p-4">
                      <div className="flex items-center justify-end space-x-2">
                        <Button 
                          size="sm" 
                          variant="ghost" 
                          className="h-8 w-8 p-0 hover:bg-blue-500/10 hover:text-blue-400"
                          onClick={() => openDetailModal(user)}
                        >
                          <Eye className="h-4 w-4" />
                        </Button>
                        <Button 
                          size="sm" 
                          variant="ghost" 
                          className="h-8 w-8 p-0 hover:bg-green-500/10 hover:text-green-400"
                          onClick={() => openEditModal(user)}
                        >
                          <Edit className="h-4 w-4" />
                        </Button>
                        <Button 
                          size="sm" 
                          variant="ghost" 
                          className="h-8 w-8 p-0 hover:bg-red-500/10 hover:text-red-400"
                          onClick={() => openDeleteModal(user)}
                        >
                          <Trash2 className="h-4 w-4" />
                        </Button>
                      </div>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>

          {loading ? (
            <div className="p-12 text-center">
              <div className="inline-block h-8 w-8 animate-spin rounded-full border-4 border-solid border-orange-500 border-r-transparent"></div>
              <p className="mt-4 text-foreground/60">Kullanıcılar yükleniyor...</p>
            </div>
          ) : users.length === 0 ? (
            <div className="p-12 text-center">
              <Users className="h-16 w-16 mx-auto text-foreground/20 mb-4" />
              <p className="text-foreground/60">Kullanıcı bulunamadı</p>
            </div>
          ) : null}

          <div className="p-4 border-t border-white/5">
            <p className="text-sm text-foreground/60">
              Toplam <strong className="text-orange-400">{total}</strong> kullanıcıdan <strong className="text-blue-400">{users.length}</strong> tanesi gösteriliyor
            </p>
          </div>
        </CardContent>
      </Card>

      {/* Edit Modal */}
      {showEditModal && selectedUser && (
        <div className="fixed inset-0 bg-black/80 backdrop-blur-sm flex items-center justify-center z-50 p-4">
          <Card className="glass-dark border-white/10 w-full max-w-2xl">
            <CardHeader className="border-b border-white/5">
              <CardTitle className="flex items-center justify-between">
                <span className="flex items-center space-x-2">
                  <Edit className="h-5 w-5 text-blue-400" />
                  <span>Kullanıcı Düzenle</span>
                </span>
                <Button variant="ghost" size="sm" onClick={() => setShowEditModal(false)}>
                  <X className="h-5 w-5" />
                </Button>
              </CardTitle>
            </CardHeader>
            <CardContent className="p-6 space-y-4">
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="text-sm font-medium mb-2 block">Ad Soyad</label>
                  <Input
                    value={formData.name}
                    onChange={(e) => setFormData({...formData, name: e.target.value})}
                    className="glass-dark border-white/10"
                  />
                </div>
                <div>
                  <label className="text-sm font-medium mb-2 block">Kullanıcı Adı</label>
                  <Input
                    value={formData.username}
                    onChange={(e) => setFormData({...formData, username: e.target.value})}
                    className="glass-dark border-white/10"
                  />
                </div>
              </div>

              <div>
                <label className="text-sm font-medium mb-2 block">Email</label>
                <Input
                  type="email"
                  value={formData.email}
                  onChange={(e) => setFormData({...formData, email: e.target.value})}
                  className="glass-dark border-white/10"
                />
              </div>

              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="text-sm font-medium mb-2 block">Rol</label>
                  <select
                    value={formData.role}
                    onChange={(e) => setFormData({...formData, role: e.target.value})}
                    className="w-full glass-dark border border-white/10 rounded-lg px-4 py-2"
                  >
                    <option value="USER">Kullanıcı</option>
                    <option value="MODERATOR">Moderatör</option>
                    <option value="ADMIN">Admin</option>
                  </select>
                </div>
                <div className="flex items-center space-x-2 pt-7">
                  <input
                    type="checkbox"
                    checked={formData.verified}
                    onChange={(e) => setFormData({...formData, verified: e.target.checked})}
                    className="rounded border-white/20 bg-white/5 cursor-pointer"
                  />
                  <label className="text-sm">Doğrulanmış Kullanıcı</label>
                </div>
              </div>

              <div className="flex space-x-3 pt-4">
                <Button 
                  onClick={handleEditUser}
                  className="flex-1 bg-gradient-to-r from-blue-500 to-purple-500 hover:from-blue-600 hover:to-purple-600"
                >
                  <CheckCircle className="h-4 w-4 mr-2" />
                  Değişiklikleri Kaydet
                </Button>
                <Button variant="outline" className="glass border-white/10" onClick={() => setShowEditModal(false)}>
                  İptal
                </Button>
              </div>
            </CardContent>
          </Card>
        </div>
      )}

      {/* Delete Confirmation Modal */}
      {showDeleteModal && selectedUser && (
        <div className="fixed inset-0 bg-black/80 backdrop-blur-sm flex items-center justify-center z-50 p-4">
          <Card className="glass-dark border-red-500/30 w-full max-w-md">
            <CardHeader className="border-b border-white/5">
              <CardTitle className="flex items-center space-x-2 text-red-400">
                <AlertCircle className="h-5 w-5" />
                <span>Kullanıcı Sil</span>
              </CardTitle>
            </CardHeader>
            <CardContent className="p-6">
              <p className="text-foreground/80 mb-4">
                <strong>"{selectedUser.name}"</strong> kullanıcısını silmek istediğinizden emin misiniz?
              </p>
              <p className="text-sm text-red-400 mb-6">
                Bu işlem geri alınamaz! Kullanıcının tüm kuponları ve yorumları silinecektir.
              </p>
              <div className="flex space-x-3">
                <Button 
                  onClick={handleDeleteUser}
                  className="flex-1 bg-gradient-to-r from-red-500 to-red-600 hover:from-red-600 hover:to-red-700"
                >
                  <Trash2 className="h-4 w-4 mr-2" />
                  Evet, Sil
                </Button>
                <Button 
                  variant="outline" 
                  className="flex-1 glass border-white/10" 
                  onClick={() => setShowDeleteModal(false)}
                >
                  İptal
                </Button>
              </div>
            </CardContent>
          </Card>
        </div>
      )}

      {/* Detail Modal */}
      {showDetailModal && selectedUser && (
        <div className="fixed inset-0 bg-black/80 backdrop-blur-sm flex items-center justify-center z-50 p-4">
          <Card className="glass-dark border-white/10 w-full max-w-2xl">
            <CardHeader className="border-b border-white/5">
              <CardTitle className="flex items-center justify-between">
                <span className="flex items-center space-x-2">
                  <Eye className="h-5 w-5 text-blue-400" />
                  <span>Kullanıcı Detayları</span>
                </span>
                <Button variant="ghost" size="sm" onClick={() => setShowDetailModal(false)}>
                  <X className="h-5 w-5" />
                </Button>
              </CardTitle>
            </CardHeader>
            <CardContent className="p-6 space-y-4">
              <div className="flex items-center space-x-4 mb-6">
                <div className="h-20 w-20 rounded-full bg-gradient-to-br from-blue-500 to-purple-500 flex items-center justify-center text-white font-bold text-3xl">
                  {(selectedUser.name || selectedUser.username || 'U').charAt(0).toUpperCase()}
                </div>
                <div>
                  <h3 className="text-2xl font-bold">{selectedUser.name || 'İsimsiz'}</h3>
                  <p className="text-foreground/60">@{selectedUser.username}</p>
                  <div className="flex items-center space-x-2 mt-2">
                    {getRoleBadge(selectedUser.role)}
                    {getStatusBadge(selectedUser.verified)}
                  </div>
                </div>
              </div>

              <div className="space-y-3">
                <div className="flex justify-between">
                  <span className="text-foreground/60">Email:</span>
                  <span className="font-semibold">{selectedUser.email}</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-foreground/60">Kayıt Tarihi:</span>
                  <span>{formatDate(selectedUser.createdAt)}</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-foreground/60">Son Güncelleme:</span>
                  <span>{getRelativeTime(selectedUser.updatedAt)}</span>
                </div>
                {selectedUser.bio && (
                  <div className="pt-2 border-t border-white/5">
                    <span className="text-foreground/60 text-sm">Bio:</span>
                    <p className="text-sm mt-1">{selectedUser.bio}</p>
                  </div>
                )}
              </div>

              <div className="grid grid-cols-2 md:grid-cols-4 gap-4 pt-4 border-t border-white/5">
                <div className="text-center glass p-3 rounded-lg">
                  <p className="text-2xl font-bold text-blue-400">{selectedUser._count?.coupons || 0}</p>
                  <p className="text-xs text-foreground/60">Kupon</p>
                </div>
                <div className="text-center glass p-3 rounded-lg">
                  <p className="text-2xl font-bold text-green-400">{selectedUser.winRate?.toFixed(1) || '0.0'}%</p>
                  <p className="text-xs text-foreground/60">Başarı Oranı</p>
                </div>
                <div className="text-center glass p-3 rounded-lg">
                  <p className="text-2xl font-bold text-purple-400">{selectedUser._count?.followers || 0}</p>
                  <p className="text-xs text-foreground/60">Takipçi</p>
                </div>
                <div className="text-center glass p-3 rounded-lg">
                  <p className="text-2xl font-bold text-orange-400">{selectedUser._count?.following || 0}</p>
                  <p className="text-xs text-foreground/60">Takip</p>
                </div>
              </div>

              <Button 
                className="w-full glass border-white/10" 
                onClick={() => setShowDetailModal(false)}
              >
                Kapat
              </Button>
            </CardContent>
          </Card>
        </div>
      )}
    </div>
  )
}