'use client'

import { useEffect, useState } from 'react'
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card'
import { Badge } from '@/components/ui/badge'
import { Button } from '@/components/ui/button'
import { 
  TrendingUp,
  TrendingDown,
  Users,
  FileText,
  MessageSquare,
  Eye,
  Heart,
  Trophy,
  DollarSign,
  Calendar,
  BarChart3,
  PieChart,
  Activity,
  CheckCircle,
  XCircle,
  Clock
} from 'lucide-react'

export default function AdminIstatistiklerPage() {
  const [loading, setLoading] = useState(true)
  const [stats, setStats] = useState<any>(null)
  const [topCoupons, setTopCoupons] = useState<any[]>([])

  const loadStats = async () => {
    setLoading(true)
    try {
      const res = await fetch('/api/admin/stats', { cache: 'no-store' })
      const data = await res.json()
      
      if (data.stats) {
        setStats(data.stats)
      }
      if (data.topCoupons) {
        setTopCoupons(data.topCoupons)
      }
    } catch (error) {
      console.error('Error loading stats:', error)
    } finally {
      setLoading(false)
    }
  }

  useEffect(() => {
    loadStats()
  }, [])

  return (
    <div className="space-y-6">
      {/* Page Header */}
      <div className="flex flex-col md:flex-row justify-between items-start md:items-center gap-4">
        <div>
          <h1 className="text-3xl font-bold gradient-text mb-2">İstatistikler</h1>
          <p className="text-foreground/60">Platform performansını izle ve analiz et</p>
        </div>
        <div className="flex items-center space-x-3">
          <Button 
            variant="outline" 
            className="glass border-white/10"
            onClick={loadStats}
          >
            <Activity className="h-4 w-4 mr-2" />
            Yenile
          </Button>
        </div>
      </div>

      {loading ? (
        <div className="p-12 text-center">
          <div className="inline-block h-8 w-8 animate-spin rounded-full border-4 border-solid border-orange-500 border-r-transparent"></div>
          <p className="mt-4 text-foreground/60">İstatistikler yükleniyor...</p>
        </div>
      ) : stats ? (
        <>
          {/* Main Stats */}
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
            <Card className="glass-dark border-white/5 hover:border-white/10 transition-all group">
              <CardContent className="p-6">
                <div className="flex items-center justify-between mb-4">
                  <div className="h-12 w-12 rounded-xl bg-blue-500/10 border border-blue-500/30 flex items-center justify-center group-hover:scale-110 transition-transform">
                    <Users className="h-6 w-6 text-blue-400" />
                  </div>
                </div>
                <p className="text-foreground/60 text-sm mb-1">Toplam Kullanıcı</p>
                <p className="text-3xl font-bold">{stats.totalUsers}</p>
                <p className="text-xs text-green-400 mt-2">+{stats.recentUsers} son 7 günde</p>
              </CardContent>
            </Card>

            <Card className="glass-dark border-white/5 hover:border-white/10 transition-all group">
              <CardContent className="p-6">
                <div className="flex items-center justify-between mb-4">
                  <div className="h-12 w-12 rounded-xl bg-purple-500/10 border border-purple-500/30 flex items-center justify-center group-hover:scale-110 transition-transform">
                    <FileText className="h-6 w-6 text-purple-400" />
                  </div>
                </div>
                <p className="text-foreground/60 text-sm mb-1">Toplam Kupon</p>
                <p className="text-3xl font-bold">{stats.totalCoupons}</p>
              </CardContent>
            </Card>

            <Card className="glass-dark border-white/5 hover:border-white/10 transition-all group">
              <CardContent className="p-6">
                <div className="flex items-center justify-between mb-4">
                  <div className="h-12 w-12 rounded-xl bg-green-500/10 border border-green-500/30 flex items-center justify-center group-hover:scale-110 transition-transform">
                    <MessageSquare className="h-6 w-6 text-green-400" />
                  </div>
                </div>
                <p className="text-foreground/60 text-sm mb-1">Toplam Yorum</p>
                <p className="text-3xl font-bold">{stats.totalComments}</p>
              </CardContent>
            </Card>

            <Card className="glass-dark border-white/5 hover:border-white/10 transition-all group">
              <CardContent className="p-6">
                <div className="flex items-center justify-between mb-4">
                  <div className="h-12 w-12 rounded-xl bg-orange-500/10 border border-orange-500/30 flex items-center justify-center group-hover:scale-110 transition-transform">
                    <Trophy className="h-6 w-6 text-orange-400" />
                  </div>
                </div>
                <p className="text-foreground/60 text-sm mb-1">Başarı Oranı</p>
                <p className="text-3xl font-bold">{stats.successRate}%</p>
              </CardContent>
            </Card>
          </div>
        </>
      ) : null}

      {/* Coupon Status Distribution */}
      {!loading && stats && (
        <div className="grid grid-cols-1 lg:grid-cols-3 gap-4">
          <Card className="glass-dark border-white/5">
            <CardContent className="p-6">
              <div className="flex items-center justify-between mb-4">
                <div className="h-12 w-12 rounded-xl bg-green-500/10 border border-green-500/30 flex items-center justify-center">
                  <CheckCircle className="h-6 w-6 text-green-400" />
                </div>
              </div>
              <p className="text-foreground/60 text-sm mb-1">Kazanan Kuponlar</p>
              <p className="text-3xl font-bold text-green-400">{stats.wonCoupons}</p>
              <div className="mt-3 h-2 bg-white/5 rounded-full overflow-hidden">
                <div 
                  className="h-full bg-gradient-to-r from-green-500 to-emerald-500"
                  style={{ width: `${stats.totalCoupons > 0 ? (stats.wonCoupons / stats.totalCoupons * 100) : 0}%` }}
                />
              </div>
            </CardContent>
          </Card>

          <Card className="glass-dark border-white/5">
            <CardContent className="p-6">
              <div className="flex items-center justify-between mb-4">
                <div className="h-12 w-12 rounded-xl bg-red-500/10 border border-red-500/30 flex items-center justify-center">
                  <XCircle className="h-6 w-6 text-red-400" />
                </div>
              </div>
              <p className="text-foreground/60 text-sm mb-1">Kaybeden Kuponlar</p>
              <p className="text-3xl font-bold text-red-400">{stats.lostCoupons}</p>
              <div className="mt-3 h-2 bg-white/5 rounded-full overflow-hidden">
                <div 
                  className="h-full bg-gradient-to-r from-red-500 to-rose-500"
                  style={{ width: `${stats.totalCoupons > 0 ? (stats.lostCoupons / stats.totalCoupons * 100) : 0}%` }}
                />
              </div>
            </CardContent>
          </Card>

          <Card className="glass-dark border-white/5">
            <CardContent className="p-6">
              <div className="flex items-center justify-between mb-4">
                <div className="h-12 w-12 rounded-xl bg-yellow-500/10 border border-yellow-500/30 flex items-center justify-center">
                  <Clock className="h-6 w-6 text-yellow-400" />
                </div>
              </div>
              <p className="text-foreground/60 text-sm mb-1">Bekleyen Kuponlar</p>
              <p className="text-3xl font-bold text-yellow-400">{stats.pendingCoupons}</p>
              <div className="mt-3 h-2 bg-white/5 rounded-full overflow-hidden">
                <div 
                  className="h-full bg-gradient-to-r from-yellow-500 to-amber-500"
                  style={{ width: `${stats.totalCoupons > 0 ? (stats.pendingCoupons / stats.totalCoupons * 100) : 0}%` }}
                />
              </div>
            </CardContent>
          </Card>
        </div>
      )}

      {/* Top Coupons */}
      {!loading && topCoupons.length > 0 && (
        <Card className="glass-dark border-white/5">
          <CardHeader className="border-b border-white/5">
            <CardTitle className="flex items-center space-x-2">
              <Trophy className="h-5 w-5 text-yellow-400" />
              <span>En Popüler Kuponlar (Görüntülenmeye Göre)</span>
            </CardTitle>
          </CardHeader>
          <CardContent className="p-6">
            <div className="space-y-3">
              {topCoupons.map((coupon, index) => (
                <div key={coupon.id} className="flex items-center justify-between p-4 glass rounded-lg border border-white/5 hover:border-white/10 transition-all group">
                  <div className="flex items-center space-x-4">
                    <div className={`h-10 w-10 rounded-lg flex items-center justify-center font-bold text-lg
                      ${index === 0 ? 'bg-yellow-500 text-black' : ''}
                      ${index === 1 ? 'bg-gray-400 text-black' : ''}
                      ${index === 2 ? 'bg-orange-600 text-white' : ''}
                      ${index > 2 ? 'bg-white/10 text-foreground/70' : ''}
                    `}>
                      {index + 1}
                    </div>
                    <div className="flex-1 min-w-0">
                      <p className="font-semibold group-hover:text-blue-400 transition truncate">{coupon.title}</p>
                      <div className="flex items-center space-x-2 mt-1">
                        <p className="text-xs text-foreground/60">@{coupon.user}</p>
                        <Badge className={
                          coupon.status === 'WON' ? 'bg-green-500/20 border-green-500/30 text-green-400 text-xs' :
                          coupon.status === 'LOST' ? 'bg-red-500/20 border-red-500/30 text-red-400 text-xs' :
                          'bg-yellow-500/20 border-yellow-500/30 text-yellow-400 text-xs'
                        }>
                          {coupon.status}
                        </Badge>
                      </div>
                    </div>
                  </div>
                  <div className="flex items-center space-x-6 text-sm">
                    <div className="flex items-center space-x-1">
                      <Eye className="h-4 w-4 text-blue-400" />
                      <span className="font-semibold">{coupon.views}</span>
                    </div>
                    <div className="flex items-center space-x-1">
                      <Heart className="h-4 w-4 text-red-400" />
                      <span>{coupon.likes}</span>
                    </div>
                    <div className="flex items-center space-x-1">
                      <MessageSquare className="h-4 w-4 text-green-400" />
                      <span>{coupon.comments}</span>
                    </div>
                  </div>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>
      )}
    </div>
  )
}


