'use client'

import { useState, useEffect } from 'react'
import { useRouter } from 'next/navigation'
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card'
import { Button } from '@/components/ui/button'
import { Input } from '@/components/ui/input'
import { Badge } from '@/components/ui/badge'
import {
  BookOpen,
  Plus,
  Search,
  Edit,
  Trash2,
  Eye,
  Calendar,
  CheckCircle,
  XCircle
} from 'lucide-react'
import Link from 'next/link'

interface BlogPost {
  id: string
  title: string
  slug: string
  excerpt: string
  category: string
  author: string
  published: boolean
  views: number
  createdAt: string
  updatedAt: string
}

export default function AdminBlogPage() {
  const router = useRouter()
  const [posts, setPosts] = useState<BlogPost[]>([])
  const [loading, setLoading] = useState(true)
  const [search, setSearch] = useState('')
  const [deleting, setDeleting] = useState<string | null>(null)

  useEffect(() => {
    loadPosts()
  }, [search])

  const loadPosts = async () => {
    setLoading(true)
    try {
      const params = new URLSearchParams()
      if (search) params.append('search', search)
      
      const res = await fetch(`/api/admin/blog?${params}`)
      if (res.ok) {
        const data = await res.json()
        setPosts(data.posts || [])
      }
    } catch (error) {
      console.error('Blog yazıları yüklenirken hata:', error)
    } finally {
      setLoading(false)
    }
  }

  const handleDelete = async (id: string) => {
    if (!confirm('Bu blog yazısını silmek istediğinizden emin misiniz?')) return

    setDeleting(id)
    try {
      const res = await fetch(`/api/admin/blog/${id}`, {
        method: 'DELETE',
      })

      if (res.ok) {
        setPosts(posts.filter(p => p.id !== id))
      } else {
        alert('Blog yazısı silinemedi')
      }
    } catch (error) {
      console.error('Blog yazısı silinirken hata:', error)
      alert('Bir hata oluştu')
    } finally {
      setDeleting(null)
    }
  }

  return (
    <div className="p-8 space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div className="flex items-center space-x-3">
          <BookOpen className="h-8 w-8 text-green-400" />
          <div>
            <h1 className="text-3xl font-bold">Blog Yönetimi</h1>
            <p className="text-foreground/60">Blog yazılarını yönet</p>
          </div>
        </div>
        <Link href="/admin/blog/yeni">
          <Button className="bg-gradient-to-r from-green-500 to-yellow-400 hover:from-green-600 hover:to-yellow-500 text-black font-bold">
            <Plus className="h-4 w-4 mr-2" />
            Yeni Blog Yazısı
          </Button>
        </Link>
      </div>

      {/* Search */}
      <Card className="glass-dark border-white/10">
        <CardContent className="p-6">
          <div className="relative">
            <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-5 w-5 text-foreground/50" />
            <Input
              type="text"
              placeholder="Blog yazısı ara..."
              value={search}
              onChange={(e) => setSearch(e.target.value)}
              className="pl-10 glass border-white/10 focus:border-green-500/50"
            />
          </div>
        </CardContent>
      </Card>

      {/* Stats */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-6">
        <Card className="glass-dark border-white/10">
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-foreground/60">Toplam Yazı</p>
                <p className="text-3xl font-bold text-white">{posts.length}</p>
              </div>
              <BookOpen className="h-10 w-10 text-green-400" />
            </div>
          </CardContent>
        </Card>

        <Card className="glass-dark border-white/10">
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-foreground/60">Yayınlanan</p>
                <p className="text-3xl font-bold text-green-400">
                  {posts.filter(p => p.published).length}
                </p>
              </div>
              <CheckCircle className="h-10 w-10 text-green-400" />
            </div>
          </CardContent>
        </Card>

        <Card className="glass-dark border-white/10">
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-foreground/60">Taslak</p>
                <p className="text-3xl font-bold text-yellow-400">
                  {posts.filter(p => !p.published).length}
                </p>
              </div>
              <XCircle className="h-10 w-10 text-yellow-400" />
            </div>
          </CardContent>
        </Card>

        <Card className="glass-dark border-white/10">
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-foreground/60">Toplam Görüntülenme</p>
                <p className="text-3xl font-bold text-blue-400">
                  {posts.reduce((sum, p) => sum + p.views, 0)}
                </p>
              </div>
              <Eye className="h-10 w-10 text-blue-400" />
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Blog Posts List */}
      <Card className="glass-dark border-white/10">
        <CardHeader>
          <CardTitle>Blog Yazıları</CardTitle>
        </CardHeader>
        <CardContent>
          {loading ? (
            <div className="text-center py-12">
              <div className="inline-block h-8 w-8 animate-spin rounded-full border-4 border-solid border-green-400 border-r-transparent"></div>
              <p className="mt-4 text-foreground/60">Yükleniyor...</p>
            </div>
          ) : posts.length === 0 ? (
            <div className="text-center py-12">
              <BookOpen className="h-16 w-16 mx-auto text-foreground/30 mb-4" />
              <p className="text-foreground/60">Henüz blog yazısı yok</p>
            </div>
          ) : (
            <div className="space-y-4">
              {posts.map((post) => (
                <div
                  key={post.id}
                  className="glass p-6 rounded-lg border border-white/5 hover:border-green-500/30 transition-all"
                >
                  <div className="flex items-start justify-between">
                    <div className="flex-1">
                      <div className="flex items-center space-x-3 mb-2">
                        <h3 className="text-xl font-bold">{post.title}</h3>
                        <Badge
                          className={
                            post.published
                              ? 'bg-green-500/20 text-green-400 border-green-500/30'
                              : 'bg-yellow-500/20 text-yellow-400 border-yellow-500/30'
                          }
                        >
                          {post.published ? 'Yayınlandı' : 'Taslak'}
                        </Badge>
                        <Badge className="bg-purple-500/20 text-purple-400 border-purple-500/30">
                          {post.category}
                        </Badge>
                      </div>
                      <p className="text-sm text-foreground/70 mb-3 line-clamp-2">
                        {post.excerpt}
                      </p>
                      <div className="flex items-center space-x-4 text-xs text-foreground/60">
                        <div className="flex items-center space-x-1">
                          <Calendar className="h-3 w-3" />
                          <span>{new Date(post.createdAt).toLocaleDateString('tr-TR')}</span>
                        </div>
                        <div className="flex items-center space-x-1">
                          <Eye className="h-3 w-3" />
                          <span>{post.views} görüntülenme</span>
                        </div>
                        <span>Yazar: {post.author}</span>
                      </div>
                    </div>

                    <div className="flex items-center space-x-2 ml-4">
                      <Link href={`/admin/blog/${post.id}`}>
                        <Button
                          size="sm"
                          className="bg-blue-500/20 hover:bg-blue-500/30 text-blue-400 border border-blue-500/30"
                        >
                          <Edit className="h-4 w-4" />
                        </Button>
                      </Link>
                      <Button
                        size="sm"
                        onClick={() => handleDelete(post.id)}
                        disabled={deleting === post.id}
                        className="bg-red-500/20 hover:bg-red-500/30 text-red-400 border border-red-500/30"
                      >
                        <Trash2 className="h-4 w-4" />
                      </Button>
                    </div>
                  </div>
                </div>
              ))}
            </div>
          )}
        </CardContent>
      </Card>
    </div>
  )
}



