import Hero from '@/components/Hero'
import FeaturedCoupons from '@/components/FeaturedCoupons'
import { Button } from '@/components/ui/button'
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card'
import { Avatar, AvatarFallback, AvatarImage } from '@/components/ui/avatar'
import { Badge } from '@/components/ui/badge'
import { TrendingUp, Flame, Award, ArrowRight, Zap, Shield, BarChart3, Crown, Trophy, Bot, Target, Megaphone } from 'lucide-react'
import Link from 'next/link'
import { prisma } from '@/lib/db'

async function getFeaturedCoupons() {
  try {
    const res = await fetch(`/api/kuponlar?limit=4`, { cache: 'no-store' })
    if (!res.ok) return []
    const data = await res.json()
    return data.coupons || []
  } catch (error) {
    console.error('Kuponlar yüklenirken hata:', error)
    return []
  }
}

async function getTopPredictors() {
  // Sistem botları - Profesyonel
  return [
    {
      id: 'admin-bot',
      username: 'admin',
      name: 'Admin',
      displayName: 'Platform Yöneticisi',
      avatar: null,
      winRate: 95,
      totalCoupons: 1250,
      followers: 5420,
      isBot: true,
      botType: 'admin',
      badge: 'Yönetici'
    },
    {
      id: 'ai-bot',
      username: 'yapay-zeka',
      name: 'Yapay Zeka',
      displayName: 'AI Tahmin Motoru',
      avatar: null,
      winRate: 88,
      totalCoupons: 3500,
      followers: 8930,
      isBot: true,
      botType: 'ai',
      badge: 'AI Sistem'
    },
    {
      id: 'sports-bot',
      username: 'spor-botu',
      name: 'Spor Botu',
      displayName: 'Spor Analiz Uzmanı',
      avatar: null,
      winRate: 82,
      totalCoupons: 2100,
      followers: 4250,
      isBot: true,
      botType: 'sports',
      badge: 'Analiz'
    },
    {
      id: 'ads-bot',
      username: 'reklam-botu',
      name: 'Reklam Botu',
      displayName: 'Sponsorlu İçerik',
      avatar: null,
      winRate: 75,
      totalCoupons: 890,
      followers: 2340,
      isBot: true,
      botType: 'promo',
      badge: 'Sponsor'
    }
  ]
}

// Dinamik sayfa - her zaman güncel veri
export const dynamic = 'force-dynamic'
export const revalidate = 0

// SEO Metadata
export async function generateMetadata() {
  return {
    title: 'İddaa Tahminleri - Banko İddaa Tahmin | Bugünkü Maç Tahminleri ve Kupon Önerileri',
    description: 'Güncel iddaa tahminleri, banko iddaa tahmini, bugünkü iddaa maç tahminleri ve analizleri. Ücretsiz iddaa kupon önerileri, yüksek oranlı tahminler ve Süper Lig analiz yorumları. Profesyonel bahis tahmini sitesi.',
    keywords: 'iddaa tahminleri, banko iddaa tahmini, iddaa maç tahminleri bugün, bugünkü iddaa tahminleri, haftalık iddaa kupon önerileri, banko kupon iddaa, yüksek oranlı iddaa tahmini, analizli iddaa tahminleri',
    alternates: {
      canonical: 'https://iddaasohbet.com',
    },
  }
}

export default async function Home() {
  const [featuredCoupons, topPredictors] = await Promise.all([
    getFeaturedCoupons(),
    getTopPredictors(),
  ])

  // Schema.org JSON-LD for SEO
  const schemaOrg = {
    '@context': 'https://schema.org',
    '@type': 'WebSite',
    name: 'İddaaSohbet',
    description: 'Güncel iddaa tahminleri, banko iddaa tahmini ve bugünkü maç tahminleri platformu',
    url: 'https://iddaasohbet.com',
    potentialAction: {
      '@type': 'SearchAction',
      target: 'https://iddaasohbet.com/kuponlar?q={search_term_string}',
      'query-input': 'required name=search_term_string',
    },
    publisher: {
      '@type': 'Organization',
      name: 'İddaaSohbet',
      logo: {
        '@type': 'ImageObject',
        url: 'https://iddaasohbet.com/logo.png',
      },
    },
  }

  return (
    <>
      {/* SEO - Schema.org JSON-LD */}
      <script
        type="application/ld+json"
        dangerouslySetInnerHTML={{ __html: JSON.stringify(schemaOrg) }}
      />
      
    <div className="min-h-screen">
      {/* Hero Section */}
      <Hero />

      {/* Featured Coupons */}
      <section className="container mx-auto px-4 py-16" aria-labelledby="featured-coupons">
        <div className="flex items-center justify-between mb-8">
          <div className="flex items-center space-x-3">
            <Flame className="h-8 w-8 text-orange-500 animate-pulse" />
            <h2 id="featured-coupons" className="text-3xl font-bold gradient-text">Güncel İddaa Kupon Önerileri ve Tahminler</h2>
          </div>
          <Link href="/kuponlar">
            <Button variant="ghost" className="hover:text-green-400 hover:bg-white/5">
              Tümünü Gör <ArrowRight className="ml-2 h-4 w-4" />
            </Button>
          </Link>
        </div>

        <FeaturedCoupons />
      </section>

      {/* Top Predictors */}
      <section className="container mx-auto px-4 py-16 bg-gradient-to-b from-transparent via-green-500/5 to-transparent" aria-labelledby="top-predictors">
        <div className="flex items-center justify-between mb-8">
          <div className="flex items-center space-x-3">
            <Crown className="h-8 w-8 text-yellow-400 animate-pulse" />
            <h2 id="top-predictors" className="text-3xl font-bold gradient-text">Profesyonel İddaa Tahmin Uzmanları</h2>
          </div>
          <Link href="/tahmincilar">
            <Button variant="ghost" className="hover:text-green-400 hover:bg-white/5">
              Tümünü Gör <ArrowRight className="ml-2 h-4 w-4" />
            </Button>
          </Link>
        </div>

        {topPredictors.length > 0 ? (
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
            {topPredictors.map((predictor, index) => (
              <Link key={predictor.id} href={`/profil/${predictor.username}`}>
                <Card className="glass-dark border-white/10 hover:border-green-500/50 transition-all duration-300 group cursor-pointer relative overflow-hidden">
                  {index === 0 && (
                    <div className="absolute top-4 right-4 z-10">
                      <Crown className="h-6 w-6 text-yellow-400 animate-pulse" />
                    </div>
                  )}
                  <CardHeader className="text-center pb-4">
                    <div className="relative inline-block mx-auto mb-4">
                      <Avatar className={`h-20 w-20 border-4 group-hover:border-yellow-400/50 transition-all ${
                        predictor.botType === 'admin' ? 'border-yellow-400/50' :
                        predictor.botType === 'ai' ? 'border-blue-400/50' :
                        predictor.botType === 'sports' ? 'border-green-500/50' :
                        'border-purple-400/50'
                      }`}>
                        {predictor.avatar ? (
                          <AvatarImage src={predictor.avatar} alt={predictor.name || 'User'} />
                        ) : (
                          <AvatarFallback className={`text-2xl font-bold ${
                            predictor.botType === 'admin' ? 'bg-gradient-to-br from-yellow-400 to-orange-500 text-black' :
                            predictor.botType === 'ai' ? 'bg-gradient-to-br from-blue-500 to-cyan-500 text-white' :
                            predictor.botType === 'sports' ? 'bg-gradient-to-br from-green-500 to-emerald-600 text-white' :
                            'bg-gradient-to-br from-purple-500 to-pink-500 text-white'
                          }`}>
                            {predictor.botType === 'admin' && <Crown className="h-10 w-10" />}
                            {predictor.botType === 'ai' && <Bot className="h-10 w-10" />}
                            {predictor.botType === 'sports' && <Target className="h-10 w-10" />}
                            {predictor.botType === 'promo' && <Megaphone className="h-10 w-10" />}
                          </AvatarFallback>
                        )}
                      </Avatar>
                      <div className="absolute -bottom-2 left-1/2 -translate-x-1/2">
                        <Badge className={`font-bold border-0 ${
                          predictor.botType === 'admin' ? 'bg-gradient-to-r from-yellow-400 to-orange-500 text-black' :
                          predictor.botType === 'ai' ? 'bg-gradient-to-r from-blue-500 to-cyan-500 text-white' :
                          predictor.botType === 'sports' ? 'bg-gradient-to-r from-green-500 to-emerald-600 text-white' :
                          'bg-gradient-to-r from-purple-500 to-pink-500 text-white'
                        }`}>
                          {predictor.badge || `#${index + 1}`}
                        </Badge>
                      </div>
                    </div>
                    <CardTitle className="text-lg">{predictor.displayName || predictor.name}</CardTitle>
                    <p className="text-sm text-foreground/60">@{predictor.username}</p>
                  </CardHeader>
                  <CardContent className="space-y-3">
                    <div className="flex items-center justify-between p-3 glass rounded-lg">
                      <span className="text-sm text-foreground/70">Başarı Oranı</span>
                      <span className="text-lg font-bold text-green-400">{predictor.winRate}%</span>
                    </div>
                    <div className="flex items-center justify-between p-3 glass rounded-lg">
                      <span className="text-sm text-foreground/70">Toplam Kupon</span>
                      <span className="text-lg font-bold">{predictor.totalCoupons}</span>
                    </div>
                    <div className="flex items-center justify-between p-3 glass rounded-lg">
                      <span className="text-sm text-foreground/70">Takipçi</span>
                      <span className="text-lg font-bold text-yellow-400">{predictor.followers}</span>
                    </div>
                  </CardContent>
                </Card>
              </Link>
            ))}
          </div>
        ) : (
          <Card className="glass-dark border-white/10 p-12 text-center">
            <div className="flex flex-col items-center space-y-4">
              <Crown className="h-16 w-16 text-foreground/30" />
              <div>
                <h3 className="text-xl font-semibold mb-2">Henüz Tahminçi Yok</h3>
                <p className="text-foreground/60">
                  Kayıt ol, kupon paylaş ve sen de tahminçi ol!
                </p>
              </div>
            </div>
          </Card>
        )}
      </section>

      {/* Stats Section */}
      <section className="container mx-auto px-4 py-16" aria-labelledby="features">
        <h2 id="features" className="sr-only">Platform Özellikleri</h2>
        <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
          <Card className="glass-dark border-white/10 hover:border-green-500/30 transition-all group">
            <CardContent className="p-8 text-center">
              <div className="inline-flex items-center justify-center w-16 h-16 rounded-full bg-green-500/10 mb-4 group-hover:bg-green-500/20 transition-all">
                <Zap className="h-8 w-8 text-green-400" />
              </div>
              <h3 className="text-3xl font-bold mb-2 gradient-text">Anında İddaa Tahminleri</h3>
              <p className="text-foreground/60">
                Bugünkü iddaa maç tahminlerini anında paylaşın ve güncel kupon önerilerini inceleyin
              </p>
            </CardContent>
          </Card>

          <Card className="glass-dark border-white/10 hover:border-yellow-400/30 transition-all group">
            <CardContent className="p-8 text-center">
              <div className="inline-flex items-center justify-center w-16 h-16 rounded-full bg-yellow-400/10 mb-4 group-hover:bg-yellow-400/20 transition-all">
                <Shield className="h-8 w-8 text-yellow-400" />
              </div>
              <h3 className="text-3xl font-bold mb-2 gradient-text">Ücretsiz İddaa Tahminleri</h3>
              <p className="text-foreground/60">
                Tüm iddaa tahminleri ve kupon önerileri tamamen ücretsiz. Gizli ücret yok
              </p>
            </CardContent>
          </Card>

          <Card className="glass-dark border-white/10 hover:border-green-500/30 transition-all group">
            <CardContent className="p-8 text-center">
              <div className="inline-flex items-center justify-center w-16 h-16 rounded-full bg-green-500/10 mb-4 group-hover:bg-green-500/20 transition-all">
                <BarChart3 className="h-8 w-8 text-green-400" />
              </div>
              <h3 className="text-3xl font-bold mb-2 gradient-text">Analizli İddaa Tahminleri</h3>
              <p className="text-foreground/60">
                Detaylı istatistikler ve analizlerle başarı oranlarını artırın
              </p>
            </CardContent>
          </Card>
        </div>
      </section>

      {/* CTA Section */}
      <section className="container mx-auto px-4 py-16">
        <Card className="glass-dark border-white/10 relative overflow-hidden">
          <div className="absolute inset-0 bg-gradient-to-r from-green-500/10 to-yellow-400/10"></div>
          <CardContent className="p-12 text-center relative z-10">
            <Award className="h-16 w-16 text-green-400 mx-auto mb-6 animate-pulse" />
            <h2 className="text-4xl font-bold mb-4 gradient-text">Hemen Başla!</h2>
            <p className="text-xl text-foreground/80 mb-8 max-w-2xl mx-auto">
              Türkiye'nin en aktif iddaa kupon paylaşım topluluğuna katıl. Ücretsiz kayıt ol, kuponlarını paylaş ve kazananlar arasına katıl!
            </p>
            <div className="flex items-center justify-center space-x-4">
              <Link href="/kayit">
                <Button size="lg" className="h-14 px-8 text-lg bg-gradient-to-r from-green-500 to-yellow-400 hover:from-green-600 hover:to-yellow-500 text-black font-bold btn-premium">
                  Ücretsiz Kayıt Ol
                </Button>
              </Link>
              <Link href="/kuponlar">
                <Button size="lg" variant="outline" className="h-14 px-8 text-lg border-white/20 hover:border-green-500/50 hover:bg-green-500/10">
                  Kuponları İncele
                </Button>
              </Link>
            </div>
          </CardContent>
        </Card>
      </section>
    </div>
    </>
  )
}