'use client'

import * as React from 'react'
import { useEffect, useRef, useState } from 'react'
import { Card } from '@/components/ui/card'
import { Button } from '@/components/ui/button'
import { Badge } from '@/components/ui/badge'
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs'
import { Radio, Trophy, RefreshCw } from 'lucide-react'

interface Fixture {
  fixture: {
    id: number
    date: string
    status: {
      long: string
      short: string
      elapsed: number | null
      extra?: number | null
    }
  }
  teams: {
    home: {
      id: number
      name: string
      logo: string
    }
    away: {
      id: number
      name: string
      logo: string
    }
  }
  goals: {
    home: number | null
    away: number | null
  }
  score: {
    halftime: {
      home: number | null
      away: number | null
    }
    fulltime: {
      home: number | null
      away: number | null
    }
  }
}

export default function CanliSkorlarPage() {
  const [activeTab, setActiveTab] = useState<'live' | 'ns' | 'ft'>('live')
  const [liveFixtures, setLiveFixtures] = useState<Fixture[]>([])
  const [todayFixtures, setTodayFixtures] = useState<Fixture[]>([])
  const [loading, setLoading] = useState(true)
  const [lastUpdate, setLastUpdate] = useState<Date>(new Date())
  const [lastLiveFetchAt, setLastLiveFetchAt] = useState<number>(Date.now())
  const [clockTick, setClockTick] = useState<number>(0)
  const elapsedSnapshotRef = useRef<Map<number, { elapsed: number, seenAt: number, phase: string }>>(new Map())
  const [previousScores, setPreviousScores] = useState<Map<number, number>>(new Map())
  const [recentGoals, setRecentGoals] = useState<Map<number, number>>(new Map()) // fixtureId -> timestamp
  const previousScoresRef = useRef<Map<number, number>>(new Map())
  const recentGoalsRef = useRef<Map<number, number>>(new Map())
  const audioCtxRef = useRef<AudioContext | null>(null)
  

  const ensureAudioContext = async () => {
    if (typeof window === 'undefined') return null
    try {
      if (!audioCtxRef.current) {
        audioCtxRef.current = new (window.AudioContext || (window as any).webkitAudioContext)()
      }
      if (audioCtxRef.current.state === 'suspended') {
        await audioCtxRef.current.resume()
      }
      return audioCtxRef.current
    } catch {
      return null
    }
  }

  const playGoalSound = async () => {
    const ctx = await ensureAudioContext()
    if (!ctx) return
    try {
      const oscillator = ctx.createOscillator()
      const gainNode = ctx.createGain()
      oscillator.type = 'triangle'
      oscillator.frequency.setValueAtTime(880, ctx.currentTime)
      gainNode.gain.setValueAtTime(0.0001, ctx.currentTime)
      gainNode.gain.exponentialRampToValueAtTime(0.25, ctx.currentTime + 0.02)
      gainNode.gain.exponentialRampToValueAtTime(0.0001, ctx.currentTime + 0.45)
      oscillator.connect(gainNode).connect(ctx.destination)
      oscillator.start()
      oscillator.stop(ctx.currentTime + 0.48)
    } catch {
      // no-op
    }
  }

  const fetchLiveScores = async () => {
    try {
      const res = await fetch(`/api/live-scores?type=live`, { cache: 'no-store' })
      const data = await res.json()
      if (Array.isArray(data?.response)) {
        const fetched: Fixture[] = data.response
        const nowTs = Date.now()
        // prune old recent goals (>20s)
        setRecentGoals((prev) => {
          const next = new Map(prev)
          for (const [id, ts] of next) {
            if (nowTs - ts > 20000) next.delete(id)
          }
          return next
        })
        // mirror to ref to avoid stale closure in intervals
        recentGoalsRef.current = new Map(recentGoalsRef.current)
        for (const [id, ts] of recentGoalsRef.current) {
          if (nowTs - ts > 20000) recentGoalsRef.current.delete(id)
        }

        // detect goals and update maps
        let goalHappened = false
        const prevScoresSnapshot = previousScoresRef.current
        const nextScores = new Map(prevScoresSnapshot)
        const indexMap = new Map<number, number>(fetched.map((fx, idx) => [fx.fixture.id, idx]))
        for (const fx of fetched) {
          const id = fx.fixture.id
          const home = fx.goals.home ?? 0
          const away = fx.goals.away ?? 0
          const sum = home + away
          const prevSum = prevScoresSnapshot.get(id)
          if (prevSum != null && sum > prevSum) {
            goalHappened = true
            setRecentGoals((prev) => {
              const next = new Map(prev)
              next.set(id, nowTs)
              return next
            })
            recentGoalsRef.current.set(id, nowTs)
          }
          nextScores.set(id, sum)
          // snapshot elapsed per fixture (refresh every fetch to stay in sync)
          const el = typeof fx.fixture.status.elapsed === 'number' ? fx.fixture.status.elapsed : null
          if (el != null) {
            const phase = fx.fixture.status.short
            elapsedSnapshotRef.current.set(id, {
              elapsed: el,
              seenAt: nowTs,
              phase,
            })
          }
        }

        // sort: recently scored first (by most recent), then keep original order
        const recentMap = new Map(recentGoalsRef.current)
        const sorted = [...fetched].sort((a, b) => {
          const at = recentMap.get(a.fixture.id) ?? 0
          const bt = recentMap.get(b.fixture.id) ?? 0
          if (at !== bt) return bt - at
          return (indexMap.get(a.fixture.id) ?? 0) - (indexMap.get(b.fixture.id) ?? 0)
        })

        setLiveFixtures(sorted)
        setPreviousScores(nextScores)
        previousScoresRef.current = nextScores
        setLastUpdate(new Date())
        setLastLiveFetchAt(Date.now())

        if (goalHappened) void playGoalSound()
      }
    } catch (error) {
      console.error('Failed to fetch live scores:', error)
    }
  }

  const fetchTodayScores = async () => {
    try {
      const res = await fetch(`/api/live-scores?type=today`, { cache: 'no-store' })
      const data = await res.json()
      if (Array.isArray(data?.response)) {
        setTodayFixtures(data.response)
        setLastUpdate(new Date())
      }
    } catch (error) {
      console.error('Failed to fetch today scores:', error)
    }
  }

  useEffect(() => {
    // Unlock audio context on first interaction to allow autoplay-like behavior
    const onAnyInput = async () => {
      await ensureAudioContext()
      window.removeEventListener('pointerdown', onAnyInput)
      window.removeEventListener('keydown', onAnyInput)
    }
    window.addEventListener('pointerdown', onAnyInput)
    window.addEventListener('keydown', onAnyInput)

    ;(async () => {
      setLoading(true)
      await Promise.all([fetchLiveScores(), fetchTodayScores()])
      setLoading(false)
    })()
    // Refresh live via API every 15s (reduce lag)
    const liveId = setInterval(() => { void fetchLiveScores() }, 15000)
    // Local clock tick to advance elapsed between fetches (every 5s)
    const tickId = setInterval(() => setClockTick((v) => v + 1), 5000)
    // Refresh today every 10 minutes
    const todayId = setInterval(fetchTodayScores, 600000)
    return () => {
      clearInterval(liveId)
      clearInterval(todayId)
      clearInterval(tickId)
      window.removeEventListener('pointerdown', onAnyInput)
      window.removeEventListener('keydown', onAnyInput)
    }
  }, [])

  // Drop goal highlights after 20s to stop blinking
  useEffect(() => {
    const id = setInterval(() => {
      setRecentGoals((prev) => {
        const nowTs = Date.now()
        let changed = false
        const next = new Map(prev)
        for (const [fid, ts] of next) {
          if (nowTs - ts > 20000) {
            next.delete(fid)
            changed = true
          }
        }
        return changed ? next : prev
      })
    }, 1000)
    return () => clearInterval(id)
  }, [])

  const getStatusBadge = (status: string) => {
    switch (status) {
      case 'FT':
        return <Badge className="bg-green-500/20 text-green-400 border-green-500/30">MS</Badge>
      case '1H':
      case '2H':
      case 'HT':
        return <Badge className="bg-red-500/20 text-red-400 border-red-500/30 animate-pulse">CANLI</Badge>
      case 'NS':
        return <Badge className="bg-yellow-500/20 text-yellow-400 border-yellow-500/30">Başlamadı</Badge>
      default:
        return <Badge className="bg-white/10 text-white/60">{status}</Badge>
    }
  }

  // kickoff time chip removed for stability; will re-add after QA

  const getDisplayedElapsed = (fx: Fixture): { elapsed: number; extra?: number } | null => {
    const phase = fx?.fixture?.status?.short
    const base = typeof fx?.fixture?.status?.elapsed === 'number' ? fx.fixture.status.elapsed : null
    const extra = typeof fx?.fixture?.status?.extra === 'number' ? fx.fixture.status.extra : null
    const livePhases = new Set(['1H', '2H', 'ET', 'LIVE'])
    if (!livePhases.has(phase)) return base != null ? { elapsed: base, extra: extra ?? undefined } : null
    
    // Use per-fixture snapshot: API's elapsed + time since last fetch
    const snap = elapsedSnapshotRef.current.get(fx.fixture.id)
    if (snap && typeof snap.elapsed === 'number' && snap.phase === phase) {
      const deltaSec = Math.max(0, Math.floor((Date.now() - snap.seenAt) / 1000))
      const addMin = Math.floor(deltaSec / 60)
      const computedElapsed = snap.elapsed + addMin
      
      // If extra time exists, return both
      if (extra != null && extra > 0) {
        return { elapsed: computedElapsed, extra }
      }
      return { elapsed: computedElapsed }
    }
    
    // Fallback: if no snapshot yet, return server elapsed
    if (base != null) {
      return { elapsed: base, extra: extra ?? undefined }
    }
    return null
  }

  return (
    <div className="min-h-screen py-12">
      <div className="container mx-auto px-4">
        {/* Header */}
        <div className="mb-8 flex flex-col md:flex-row md:items-center md:justify-between gap-4">
          <div>
            <div className="flex items-center gap-3 mb-2">
              <div className="relative">
                <Radio className="h-10 w-10 text-red-500 animate-pulse" />
                <div className="absolute inset-0 bg-red-500 blur-xl opacity-40"></div>
              </div>
              <h1 className="text-4xl font-bold gradient-text">Canlı Skorlar</h1>
              <div className="flex h-3 w-3">
                <span className="animate-ping absolute inline-flex h-3 w-3 rounded-full bg-red-400 opacity-75"></span>
                <span className="relative inline-flex rounded-full h-3 w-3 bg-red-500"></span>
              </div>
            </div>
             <p className="text-foreground/60">Tüm dünyadan canlı maç skorları</p>
            <p className="text-sm text-foreground/40 mt-1">
              Son güncelleme: {lastUpdate.toLocaleTimeString('tr-TR')}
            </p>
          </div>
          <div className="flex gap-3">
            <Button
              onClick={async () => {
                await Promise.all([fetchLiveScores(), fetchTodayScores()])
              }}
              disabled={loading}
              variant="outline"
              className="border-white/10 hover:border-blue-500/50"
            >
              <RefreshCw className={`h-4 w-4 ${loading ? 'animate-spin' : ''}`} />
            </Button>
          </div>
        </div>

        {/* Tabs */}
        <Tabs value={activeTab} onValueChange={(v) => setActiveTab(v as any)} className="w-full">
          <TabsList className="bg-white/5 border border-white/10">
            <TabsTrigger value="live" className="data-[state=active]:bg-red-500/10 data-[state=active]:text-red-400">Canlı</TabsTrigger>
            <TabsTrigger value="ns" className="data-[state=active]:bg-yellow-500/10 data-[state=active]:text-yellow-400">Başlamadı</TabsTrigger>
            <TabsTrigger value="ft" className="data-[state=active]:bg-green-500/10 data-[state=active]:text-green-400">Bitti</TabsTrigger>
          </TabsList>

          {/* Matches List */}
          {loading && liveFixtures.length === 0 && todayFixtures.length === 0 ? (
          <div className="space-y-2">
            {[...Array(8)].map((_, i) => (
              <div key={i} className="h-12 w-full rounded-lg bg-white/5 animate-pulse"></div>
            ))}
          </div>
          ) : (
          <>
          <TabsContent value="live">
            {liveFixtures.length > 0 ? (
            <Card className="glass-dark border-white/10 overflow-hidden">
              <div className="divide-y divide-white/10">
                {liveFixtures.map((fx) => {
                const highlightTs = recentGoals.get(fx.fixture.id)
                const isHighlighted = typeof highlightTs === 'number' && Date.now() - highlightTs <= 20000
                return (
                <div
                  key={fx.fixture.id}
                  className={`grid grid-cols-12 items-center h-14 px-3 hover:bg-white/5 transition ${isHighlighted ? 'goal-blink' : ''}`}
                >
                  {/* Status */}
                  <div className="col-span-2 flex items-center gap-2">
                    {getStatusBadge(fx.fixture.status.short)}
                    {(() => {
                      const elapsedData = getDisplayedElapsed(fx)
                      if (!elapsedData) return null
                      const display = elapsedData.extra != null && elapsedData.extra > 0
                        ? `${elapsedData.elapsed}+${elapsedData.extra}`
                        : `${elapsedData.elapsed}`
                      return <span className="text-xs text-red-400 font-semibold">{display}'</span>
                    })()}
                  </div>
                  {/* Home */}
                  <div className="col-span-4 flex items-center gap-2 min-w-0">
                    <img src={fx.teams.home.logo} alt={fx.teams.home.name} className="h-5 w-5 object-contain" onError={(e) => { e.currentTarget.style.display = 'none' }} />
                    <span className="truncate">{fx.teams.home.name}</span>
                  </div>
                  {/* Score */}
                  <div className={`col-span-2 text-center font-bold ${isHighlighted ? 'score-blink' : ''}`}>
                    {(fx.goals.home ?? '-')}{' '}-{' '}{(fx.goals.away ?? '-')}
                  </div>
                  {/* Away */}
                  <div className="col-span-4 flex items-center gap-2 min-w-0 justify-end">
                    <span className="truncate text-right">{fx.teams.away.name}</span>
                    <img src={fx.teams.away.logo} alt={fx.teams.away.name} className="h-5 w-5 object-contain" onError={(e) => { e.currentTarget.style.display = 'none' }} />
                  </div>
                </div>
                )})}
              </div>
            </Card>
            ) : (
              <Card className="glass-dark border-white/10 p-12 text-center">
                <div className="flex flex-col items-center space-y-4">
                   <Trophy className="h-16 w-16 text-foreground/30" />
                   <div>
                     <h3 className="text-xl font-semibold mb-2">Canlı Maç Bulunamadı</h3>
                     <p className="text-foreground/60">Şu anda canlı maç bulunmuyor.</p>
                   </div>
                </div>
              </Card>
            )}
          </TabsContent>

          <TabsContent value="ns">
            {todayFixtures.filter((f) => f.fixture.status.short === 'NS').length > 0 ? (
            <Card className="glass-dark border-white/10 overflow-hidden">
              <div className="divide-y divide-white/10">
                {todayFixtures.filter((f) => f.fixture.status.short === 'NS').map((fx) => (
                  <div
                    key={fx.fixture.id}
                    className="grid grid-cols-12 items-center h-14 px-3 hover:bg-white/5 transition"
                  >
                    <div className="col-span-2 flex items-center gap-2">
                      {getStatusBadge(fx.fixture.status.short)}
                    </div>
                    <div className="col-span-4 flex items-center gap-2 min-w-0">
                      <img src={fx.teams.home.logo} alt={fx.teams.home.name} className="h-5 w-5 object-contain" onError={(e) => { e.currentTarget.style.display = 'none' }} />
                      <span className="truncate">{fx.teams.home.name}</span>
                    </div>
                    <div className="col-span-2 text-center font-bold">-</div>
                    <div className="col-span-4 flex items-center gap-2 min-w-0 justify-end">
                      <span className="truncate text-right">{fx.teams.away.name}</span>
                      <img src={fx.teams.away.logo} alt={fx.teams.away.name} className="h-5 w-5 object-contain" onError={(e) => { e.currentTarget.style.display = 'none' }} />
                    </div>
                  </div>
                ))}
              </div>
            </Card>
            ) : (
              <Card className="glass-dark border-white/10 p-12 text-center">
                <div className="flex flex-col items-center space-y-4">
                   <Trophy className="h-16 w-16 text-foreground/30" />
                   <div>
                     <h3 className="text-xl font-semibold mb-2">Başlamamış Maç Bulunamadı</h3>
                     <p className="text-foreground/60">Bugün için başlamamış maç yok.</p>
                   </div>
                </div>
              </Card>
            )}
          </TabsContent>

          <TabsContent value="ft">
            {todayFixtures.filter((f) => f.fixture.status.short === 'FT').length > 0 ? (
            <Card className="glass-dark border-white/10 overflow-hidden">
              <div className="divide-y divide-white/10">
                {todayFixtures.filter((f) => f.fixture.status.short === 'FT').map((fx) => (
                  <div
                    key={fx.fixture.id}
                    className="grid grid-cols-12 items-center h-14 px-3 hover:bg-white/5 transition"
                  >
                    <div className="col-span-2 flex items-center gap-2">
                      {getStatusBadge(fx.fixture.status.short)}
                    </div>
                    <div className="col-span-4 flex items-center gap-2 min-w-0">
                      <img src={fx.teams.home.logo} alt={fx.teams.home.name} className="h-5 w-5 object-contain" onError={(e) => { e.currentTarget.style.display = 'none' }} />
                      <span className="truncate">{fx.teams.home.name}</span>
                    </div>
                    <div className="col-span-2 text-center font-bold">
                      {(fx.goals.home ?? '-')}{' '}-{' '}{(fx.goals.away ?? '-')}
                    </div>
                    <div className="col-span-4 flex items-center gap-2 min-w-0 justify-end">
                      <span className="truncate text-right">{fx.teams.away.name}</span>
                      <img src={fx.teams.away.logo} alt={fx.teams.away.name} className="h-5 w-5 object-contain" onError={(e) => { e.currentTarget.style.display = 'none' }} />
                    </div>
                  </div>
                ))}
              </div>
            </Card>
            ) : (
              <Card className="glass-dark border-white/10 p-12 text-center">
                <div className="flex flex-col items-center space-y-4">
                   <Trophy className="h-16 w-16 text-foreground/30" />
                   <div>
                     <h3 className="text-xl font-semibold mb-2">Bitmiş Maç Bulunamadı</h3>
                     <p className="text-foreground/60">Bugün için bitmiş maç yok.</p>
                   </div>
                </div>
              </Card>
            )}
          </TabsContent>
          </>
          )}
        </Tabs>

        {/* Stats modal removed as requested */}
      </div>
      <style jsx>{`
        @keyframes goalFlash {
          0% { background-color: rgba(34, 197, 94, 0.10); }
          50% { background-color: rgba(34, 197, 94, 0.28); }
          100% { background-color: rgba(34, 197, 94, 0.10); }
        }
        .goal-blink { animation: goalFlash 1s ease-in-out infinite; }
        .score-blink { color: #22c55e; text-shadow: 0 0 8px rgba(34, 197, 94, 0.7); }
      `}</style>
    </div>
  )
}
