const {
  SlashCommandBuilder,
  PermissionFlagsBits,
  EmbedBuilder,
  ActionRowBuilder,
  StringSelectMenuBuilder,
  ChannelType,
  ButtonBuilder,
  ButtonStyle,
  ModalBuilder,
  TextInputBuilder,
  TextInputStyle
} = require('discord.js');

const setupStates = new Map();

// ✅ SADECE BU CUSTOMID'LER GELİRSE kayıt-kur DEVREYE GİRSİN
const ALLOWED_IDS = new Set([
  // select menüler
  'kayit_kanal_sec',
  'sohbet_kanal_sec',
  'ses_kanal_sec',
  'kayitsiz_rol_sec',
  'uye_rol_sec',
  // butonlar
  'btn_kayitsiz_id',
  'btn_uye_id',
  // modallar
  'modal_kayitsiz_id',
  'modal_uye_id',
]);

module.exports = {
  data: new SlashCommandBuilder()
    .setName('kayıt-kur')
    .setDescription('Nick-only kayıt sistemini adım adım kurar (Rol ID destekli)')
    .setDefaultMemberPermissions(PermissionFlagsBits.Administrator),

  async execute(interaction) {
    const embed = new EmbedBuilder()
      .setColor('#5865F2')
      .setTitle('⚙️ Kayıt Sistemi Kurulum')
      .setDescription('**1/5** – Kayıt Kanalını Seçin')
      .setFooter({ text: 'Adım 1/5' })
      .setTimestamp();

    const channels = interaction.guild.channels.cache.filter(ch => ch.type === ChannelType.GuildText);
    if (channels.size === 0) {
      return interaction.reply({ content: '⚠️ Sunucuda metin kanalı bulunamadı!', ephemeral: true });
    }

    const options = channels.map(ch => ({ label: ch.name, value: ch.id })).slice(0, 25);

    const menu = new StringSelectMenuBuilder()
      .setCustomId('kayit_kanal_sec')
      .setPlaceholder('Kayıt kanalını seçin')
      .addOptions(options);

    setupStates.set(interaction.user.id, { guildId: interaction.guild.id, step: 1, data: {} });

    await interaction.reply({
      embeds: [embed],
      components: [new ActionRowBuilder().addComponents(menu)],
      ephemeral: true
    });
  },

  async handleInteraction(interaction, client) {
    // ✅ kayıt-kur'a ait olmayan menü/buton/modal ise ASLA cevap verme
    if (interaction.isStringSelectMenu() || interaction.isButton() || interaction.isModalSubmit()) {
      if (!ALLOWED_IDS.has(interaction.customId)) return;
    } else {
      return;
    }

    const state = setupStates.get(interaction.user.id);

    // ✅ yalnızca kendi interactionlarımızda state yoksa uyar
    if (!state) {
      return interaction.reply({
        content: '⚠️ Kurulum oturumu bulunamadı. `/kayıt-kur` komutunu tekrar çalıştırın.',
        ephemeral: true
      }).catch(() => {});
    }

    // ───── SELECT MENÜLER ─────
    if (interaction.isStringSelectMenu()) {
      const value = interaction.values[0];

      switch (interaction.customId) {
        case 'kayit_kanal_sec':
          state.data.kayitKanal = value;
          state.step = 2;
          return showStep2(interaction, state);

        case 'sohbet_kanal_sec':
          state.data.sohbetKanal = value;
          state.step = 3;
          return showStep3(interaction, state);

        case 'ses_kanal_sec':
          state.data.sesTeyitKanal = value;
          state.step = 4;
          return showStep4(interaction, state);

        case 'kayitsiz_rol_sec':
          state.data.kayitsizRol = value;
          state.step = 5;
          return showStep5(interaction, state);

        case 'uye_rol_sec':
          state.data.uyeRol = value;
          return finalizeSetup(interaction, state, client);
      }
    }

    // ───── BUTONLAR (ROL ID GİR) ─────
    if (interaction.isButton()) {
      if (interaction.customId === 'btn_kayitsiz_id') {
        const modal = new ModalBuilder()
          .setCustomId('modal_kayitsiz_id')
          .setTitle('Kayıtsız Rol ID Gir');

        const input = new TextInputBuilder()
          .setCustomId('role_id')
          .setLabel('Kayıtsız rolün IDsi')
          .setStyle(TextInputStyle.Short)
          .setPlaceholder('Örn: 123456789012345678')
          .setRequired(true);

        modal.addComponents(new ActionRowBuilder().addComponents(input));
        return interaction.showModal(modal);
      }

      if (interaction.customId === 'btn_uye_id') {
        const modal = new ModalBuilder()
          .setCustomId('modal_uye_id')
          .setTitle('Üye Rol ID Gir');

        const input = new TextInputBuilder()
          .setCustomId('role_id')
          .setLabel('Üye rolün IDsi')
          .setStyle(TextInputStyle.Short)
          .setPlaceholder('Örn: 123456789012345678')
          .setRequired(true);

        modal.addComponents(new ActionRowBuilder().addComponents(input));
        return interaction.showModal(modal);
      }
    }

    // ───── MODAL SUBMIT (ID KAYDET) ─────
    if (interaction.isModalSubmit()) {
      const roleId = interaction.fields.getTextInputValue('role_id')?.trim();

      const role = interaction.guild.roles.cache.get(roleId);
      if (!role) {
        return interaction.reply({
          content: '❌ Bu ID ile rol bulunamadı. Rol ID doğru mu?',
          ephemeral: true
        }).catch(() => {});
      }

      if (interaction.customId === 'modal_kayitsiz_id') {
        state.data.kayitsizRol = roleId;
        state.step = 5;

        await interaction.reply({
          content: `✅ Kayıtsız rol seçildi: **${role.name}**`,
          ephemeral: true
        }).catch(() => {});

        return showStep5AsFollowUp(interaction, state);
      }

      if (interaction.customId === 'modal_uye_id') {
        state.data.uyeRol = roleId;

        await interaction.reply({
          content: `✅ Üye rol seçildi: **${role.name}**`,
          ephemeral: true
        }).catch(() => {});

        return finalizeSetupAsFollowUp(interaction, state, client);
      }
    }
  }
};

// ─────────────── ADIMLAR ───────────────

async function showStep2(interaction, state) {
  const embed = new EmbedBuilder()
    .setColor('#5865F2')
    .setTitle('⚙️ Kayıt Sistemi Kurulum')
    .setDescription('**2/5** – Sohbet Kanalını Seçin')
    .setFooter({ text: 'Adım 2/5' })
    .setTimestamp();

  const channels = interaction.guild.channels.cache.filter(ch => ch.type === ChannelType.GuildText);
  const options = channels.map(ch => ({ label: ch.name, value: ch.id })).slice(0, 25);

  const menu = new StringSelectMenuBuilder()
    .setCustomId('sohbet_kanal_sec')
    .setPlaceholder('Sohbet kanalını seçin')
    .addOptions(options);

  await interaction.update({
    embeds: [embed],
    components: [new ActionRowBuilder().addComponents(menu)]
  });
}

async function showStep3(interaction, state) {
  const embed = new EmbedBuilder()
    .setColor('#5865F2')
    .setTitle('⚙️ Kayıt Sistemi Kurulum')
    .setDescription('**3/5** – Ses Teyit Kanalını Seçin')
    .setFooter({ text: 'Adım 3/5' })
    .setTimestamp();

  const channels = interaction.guild.channels.cache.filter(ch => ch.type === ChannelType.GuildVoice);
  const options = channels.map(ch => ({ label: ch.name, value: ch.id })).slice(0, 25);

  const menu = new StringSelectMenuBuilder()
    .setCustomId('ses_kanal_sec')
    .setPlaceholder('Ses teyit kanalını seçin')
    .addOptions(options);

  await interaction.update({
    embeds: [embed],
    components: [new ActionRowBuilder().addComponents(menu)]
  });
}

async function showStep4(interaction, state) {
  const embed = new EmbedBuilder()
    .setColor('#5865F2')
    .setTitle('⚙️ Kayıt Sistemi Kurulum')
    .setDescription('**4/5** – Kayıtsız Rolünü Seçin\nListede yoksa **Rol ID Gir** butonunu kullan.')
    .setFooter({ text: 'Adım 4/5' })
    .setTimestamp();

  const roles = interaction.guild.roles.cache
    .filter(r => r.id !== interaction.guild.id && !r.managed)
    .sort((a, b) => b.position - a.position);

  const options = roles.map(r => ({ label: r.name, value: r.id })).slice(0, 25);

  const menu = new StringSelectMenuBuilder()
    .setCustomId('kayitsiz_rol_sec')
    .setPlaceholder('Kayıtsız rolünü seçin')
    .addOptions(options.length ? options : [{ label: 'Rol bulunamadı', value: interaction.guild.id }]);

  const row1 = new ActionRowBuilder().addComponents(menu);
  const row2 = new ActionRowBuilder().addComponents(
    new ButtonBuilder().setCustomId('btn_kayitsiz_id').setLabel('Rol ID Gir').setStyle(ButtonStyle.Primary)
  );

  await interaction.update({ embeds: [embed], components: [row1, row2] });
}

async function showStep5(interaction, state) {
  const embed = new EmbedBuilder()
    .setColor('#5865F2')
    .setTitle('⚙️ Kayıt Sistemi Kurulum')
    .setDescription('**5/5** – Üye Rolünü Seçin\nListede yoksa **Rol ID Gir** butonunu kullan.')
    .setFooter({ text: 'Adım 5/5' })
    .setTimestamp();

  const roles = interaction.guild.roles.cache
    .filter(r => r.id !== interaction.guild.id && !r.managed)
    .sort((a, b) => b.position - a.position);

  const options = roles.map(r => ({ label: r.name, value: r.id })).slice(0, 25);

  const menu = new StringSelectMenuBuilder()
    .setCustomId('uye_rol_sec')
    .setPlaceholder('Üye rolünü seçin')
    .addOptions(options.length ? options : [{ label: 'Rol bulunamadı', value: interaction.guild.id }]);

  const row1 = new ActionRowBuilder().addComponents(menu);
  const row2 = new ActionRowBuilder().addComponents(
    new ButtonBuilder().setCustomId('btn_uye_id').setLabel('Rol ID Gir').setStyle(ButtonStyle.Primary)
  );

  await interaction.update({ embeds: [embed], components: [row1, row2] });
}

// Modal üzerinden reply attığımız için update değil followUp ile adım gösteriyoruz:
async function showStep5AsFollowUp(interaction, state) {
  const embed = new EmbedBuilder()
    .setColor('#5865F2')
    .setTitle('⚙️ Kayıt Sistemi Kurulum')
    .setDescription('**5/5** – Üye Rolünü Seçin\nListede yoksa **Rol ID Gir** butonunu kullan.')
    .setFooter({ text: 'Adım 5/5' })
    .setTimestamp();

  const roles = interaction.guild.roles.cache
    .filter(r => r.id !== interaction.guild.id && !r.managed)
    .sort((a, b) => b.position - a.position);

  const options = roles.map(r => ({ label: r.name, value: r.id })).slice(0, 25);

  const menu = new StringSelectMenuBuilder()
    .setCustomId('uye_rol_sec')
    .setPlaceholder('Üye rolünü seçin')
    .addOptions(options.length ? options : [{ label: 'Rol bulunamadı', value: interaction.guild.id }]);

  const row1 = new ActionRowBuilder().addComponents(menu);
  const row2 = new ActionRowBuilder().addComponents(
    new ButtonBuilder().setCustomId('btn_uye_id').setLabel('Rol ID Gir').setStyle(ButtonStyle.Primary)
  );

  return interaction.followUp({ embeds: [embed], components: [row1, row2], ephemeral: true }).catch(() => {});
}

async function finalizeSetup(interaction, state, client) {
  const ayarlar = client.loadAyarlar();
  ayarlar[state.guildId] = state.data;
  client.saveAyarlar(ayarlar);

  setupStates.delete(interaction.user.id);

  const embed = new EmbedBuilder()
    .setColor('#57f287')
    .setTitle('✅ Kurulum Tamamlandı!')
    .addFields(
      { name: '📝 Kayıt Kanalı', value: `<#${state.data.kayitKanal}>`, inline: true },
      { name: '💬 Sohbet Kanalı', value: `<#${state.data.sohbetKanal}>`, inline: true },
      { name: '🔊 Ses Teyit', value: `<#${state.data.sesTeyitKanal}>`, inline: true },
      { name: '❌ Kayıtsız Rol', value: `<@&${state.data.kayitsizRol}>`, inline: true },
      { name: '✅ Üye Rol', value: `<@&${state.data.uyeRol}>`, inline: true }
    )
    .setFooter({ text: 'Nick-only kayıt sistemi aktif!' })
    .setTimestamp();

  await interaction.update({ embeds: [embed], components: [] });
}

// Modal submitte update yapamadığımız için followUp ile bitiriyoruz:
async function finalizeSetupAsFollowUp(interaction, state, client) {
  const ayarlar = client.loadAyarlar();
  ayarlar[state.guildId] = state.data;
  client.saveAyarlar(ayarlar);

  setupStates.delete(interaction.user.id);

  const embed = new EmbedBuilder()
    .setColor('#57f287')
    .setTitle('✅ Kurulum Tamamlandı!')
    .addFields(
      { name: '📝 Kayıt Kanalı', value: `<#${state.data.kayitKanal}>`, inline: true },
      { name: '💬 Sohbet Kanalı', value: `<#${state.data.sohbetKanal}>`, inline: true },
      { name: '🔊 Ses Teyit', value: `<#${state.data.sesTeyitKanal}>`, inline: true },
      { name: '❌ Kayıtsız Rol', value: `<@&${state.data.kayitsizRol}>`, inline: true },
      { name: '✅ Üye Rol', value: `<@&${state.data.uyeRol}>`, inline: true }
    )
    .setFooter({ text: 'Nick-only kayıt sistemi aktif!' })
    .setTimestamp();

  return interaction.followUp({ embeds: [embed], components: [], ephemeral: true }).catch(() => {});
}
